/*! \file i2c.c \brief I2C interface using AVR Two-Wire Interface (TWI) hardware. */
//*****************************************************************************
//
// File Name	: 'i2c.c'
// Title			: I2C interface using AVR Two-Wire Interface (TWI) hardware
// Author			: Pascal Stang - Copyright (C) 2002-2003
// Rewriter 	: Zoltan Bali 
// Created		: 2002.06.25
// Revised		: 2004.03.22
// Version		: 0.9
// Target MCU					: Atmel AVR series
// Target Compiler			: IAR EW 3.10C
// Editor Tabs					: 4
//
// Description : I2C (pronounced "eye-squared-see") is a two-wire bidirectional
//		network designed for easy transfer of information between a wide variety
//		of intelligent devices.  Many of the Atmel AVR series processors have
//		hardware support for transmitting and receiving using an I2C-type bus.
//		In addition to the AVRs, there are thousands of other parts made by
//		manufacturers like Philips, Maxim, National, TI, etc that use I2C as
//		their primary means of communication and control.  Common device types
//		are A/D & D/A converters, temp sensors, intelligent battery monitors,
//		MP3 decoder chips, EEPROM chips, multiplexing switches, etc.
//
//		I2C uses only two wires (SDA and SCL) to communicate bidirectionally
//		between devices.	I2C is a multidrop network, meaning that you can have
//		several devices on a single bus.	Because I2C uses a 7-bit number to
//		identify which device it wants to talk to, you cannot have more than
//		127 devices on a single bus.
//
//		I2C ordinarily requires two 4.7K pull-up resistors to power (one each on
//		SDA and SCL), but for small numbers of devices (maybe 1-4), it is enough
//		to activate the internal pull-up resistors in the AVR processor.	To do
//		this, set the port pins, which correspond to the I2C pins SDA/SCL, high.
//		For example, on the mega163, sbi(PORTC, 0); sbi(PORTC, 1);.
//
//		For complete information about I2C, see the Philips Semiconductor
//		website.	They created I2C and have the largest family of devices that
//		work with I2C.
//
// Note: Many manufacturers market I2C bus devices under a different or generic
//		bus name like "Two-Wire Interface".  This is because Philips still holds
//		"I2C" as a trademark.  For example, SMBus and SMBus devices are hardware
//		compatible and closely related to I2C.	They can be directly connected
//		to an I2C bus along with other I2C devices are are generally accessed in
//		the same way as I2C devices.	SMBus is often found on modern motherboards
//		for temp sensing and other low-level control tasks.
//
//
//*****************************************************************************

#include "config.h"     //All define constant expr. and macros contain this file
#include "types.h"
#include <ina90.h>

//#include <interrupt.h>
#include <twi.h>
#include "i2c.h"
#include "uart2.h"

BYTE message_flag;

// Standard I2C bit rates are:
// 100KHz for slow speed
// 400KHz for high speed

//------------------------------------------------------------------------------------
//	Prototypes
//------------------------------------------------------------------------------------
void SM_Send16(BYTE chip_select, unsigned int byte_address,
	unsigned char* ptr, unsigned int bcount);
void SM_Send8(BYTE chip_select, unsigned char byte_address,
	unsigned char* ptr, unsigned int bcount);
void SM_Receive16(BYTE chip_select, unsigned int byte_address,
	unsigned char* ptr, unsigned int bcount);		
void SM_Receive8(BYTE chip_select, unsigned char byte_address,
	unsigned char* ptr, unsigned int bcount);
//------------------------------------------------------------------------------------
//	Variables
//------------------------------------------------------------------------------------
static	GPP_FLAG_F i2c_flags; 					      // I2C general purpose flags
static	BYTE COMMAND; 												// Holds the slave address + R/W bit for 
// use in the I2C Bus ISR.
// OR data that has just been received.
static	WORD BYTE_NUMBER; 										// Used by ISR to check what data has just been
// sent - High address byte, Low byte, or data byte
static	BYTE HIGH_ADD, LOW_ADD; 							// High & Low byte for EEPROM memory address
static	BYTE* I2C_BUFF; 											// I2C pointer to buffer to data
static	WORD I2C_COUNT; 											// Data byte counter
//static	BYTE action;													// Desired TWI action 	STO,STA,EA
#define LAST_BYTE 	i2c_flags.Bits.b0 				// Last byte send flag				
#define ADDR16			i2c_flags.Bits.b1 				// 16 bit address flag
#define ADDR16_SENT i2c_flags.Bits.b2 				// Second byte LOW_ADD sent 	 
#define REP_START 	i2c_flags.Bits.b3 				// Repeated start sent flag. Addressing folowed by master received 				
#define SM_BUSY 		i2c_flags.Bits.b4					// This bit is set when a send or receive 
// is started. It is cleared by the interupt STO condition
static volatile eI2cStateType I2cState;

// functions
void i2cInit(void)
{
	// set pull-up resistors on I2C bus pins
	sbi(PORTC, 0);	// i2c SCL on ATmega163,323,16,32,etc
	sbi(PORTC, 1);	// i2c SDA on ATmega163,323,16,32,etc
	sbi(PORTD, 0);	// i2c SCL on ATmega128,64
	sbi(PORTD, 1);	// i2c SDA on ATmega128,64

	// clear SlaveReceive and SlaveTransmit handler to null
	// set i2c bit rate to 200KHz
	i2cSetBitrate(100);
	// enable TWI (two-wire interface)
	sbi(TWCR, TWEN);
	// set state
	SM_BUSY = 0;
	// enable TWI interrupt and slave address ACK
	sbi(TWCR, TWIE);
	sbi(TWCR, TWEA);
	// outb(TWCR, (inb(TWCR)&TWCR_CMD_MASK)|BV(TWINT)|BV(TWEA));
	//	_SEI(); 	Originale enable interrupts moved to main() 
}

void i2cSetBitrate(u16 bitrateKHz)
{
	u08 bitrate_div;
	// set i2c bitrate
	// SCL freq = F_CPU/(16+2*TWBR))
#ifdef TWPS0
	// for processors with additional bitrate division (mega128)
	// SCL freq = F_CPU/(16+2*TWBR*4^TWPS)
	// set TWPS to zero
	cbi(TWSR, TWPS0);
	cbi(TWSR, TWPS1);
#endif
	// calculate bitrate division	
	bitrate_div = ((F_CPU / 1000L) / bitrateKHz);
	if (bitrate_div >= 16)
		bitrate_div = (bitrate_div - 16) / 2;
	outb(TWBR, bitrate_div);

	// 'cause the PIC so slow, it needs maximum division
	// if using a REAL mcu, delete the following line
}

void i2cSetLocalDeviceAddr(u08 deviceAddr, u08 genCallEn)
{
	// set local device address (used in slave mode only)
	outb(TWAR, ((deviceAddr & 0xFE) | (genCallEn ? 1 : 0)));
}

#pragma inline =forced 
void i2cSendStart(void)
{
	// send start condition
	outb(TWCR, (inb(TWCR) & TWCR_CMD_MASK) | BV(TWSTA));
}
#pragma inline =forced 
void i2cSendStop(void)
{
	// transmit stop condition
	// leave with TWEA on for slave receiving
	outb(TWCR, (inb(TWCR) & TWCR_CMD_MASK) | BV(TWINT) | BV(TWSTO));
}
//#pragma inline =forced 
//void i2cWaitForComplete(void)
//{
	// wait for i2c interface to complete operation
//	while (!(inb(TWCR) & BV(TWINT)));
//}
//#pragma inline =forced 
//void i2cSendByte(u08 data)
//{
	// save data to the TWDR
//	outb(TWDR, data);
	// begin send	
//	outb(TWCR, (inb(TWCR) & TWCR_CMD_MASK) | BV(TWINT));
	// Clear INT and enable TWI operation
//}
//#pragma inline =forced 
//void i2cReceiveByte(u08 ackFlag)
//{
	// begin receive over i2c
//	if (ackFlag)
//	{
		// ackFlag = TRUE: ACK the recevied data
//		outb(TWCR, (inb(TWCR) & TWCR_CMD_MASK) | BV(TWINT) | BV(TWEA));
		// Clear INT and enable TWI operation
	//}
	//else
	//{
		// ackFlag = FALSE: NACK the recevied data
	//	outb(TWCR, (inb(TWCR) & TWCR_CMD_MASK) | BV(TWINT));
		// Clear INT and enable TWI operation
	//}
//}
//#pragma inline =forced 
//u08 i2cGetReceivedByte(void)
//{
	// retieve received data byte from i2c TWDR
//	return(inb(TWDR));
//}
//#pragma inline =forced 
//u08 i2cGetStatus(void)
//{
	// retieve current i2c status from i2c TWSR
//	return(inb(TWSR));
//}


#if defined(USE_I2CEEPROM)

static void MakeChipSelect(BYTE* cs, BYTE* adh)
{
#if defined E24LC64 			// 8K chip

	*cs |= (*adh >> 4) & 0x0e;
	*adh &= 0x1f;	 			

#elif defined E24LC128		// 16K chip

	*cs |= (*adh >> 5) & 0x06;
	*adh &= 0x3f;	 			
#elif defined E24LC256		// 32K chip  	

	*cs |= (*adh >> 6) & 0x02;
	*adh &= 0x7f;	 			
#elif defined  E24LC513  	// 64K chip

	return
	//	*cs  |= (*adh)&0x00;
	//	*adh &= 0xff;	 	
#else		
#error "Invalid eeprom type or not defined !"
#endif
	return;
}
#endif


//------------------------------------------------------------------------------------
//	I2C function
//------------------------------------------------------------------------------------
// ISR when the operation is finished.				
// SMBus write function-----------------------------------------------------
// Writes a multiply byte at the specified memory location.
//
// *ptr = data byte to be written
// bcount = data byte count to write
// byte_address(16 bit) = memory location to be written into 
// chip_select = device address of EEPROM chip to be written to
#if defined(DEVICE_16)
void SM_Send16(BYTE chip_select, unsigned int byte_address,
	unsigned char* ptr, unsigned int bcount)
{
	while (SM_BUSY && inb(TWCR)&BV(TWSTO));			      // Wait for SMBus to be free.
	SM_BUSY = 1;					                            // Occupy SMBus (set to busy)
	ADDR16 = 1; 															        // 16 bit adressing mode		
	ADDR16_SENT = 0;
	HIGH_ADD = ((byte_address >> 8) & 0x00FF);	      // Upper 8 address bits
	LOW_ADD = (byte_address & 0x00FF);					      // Lower 8 address bits	
	MakeChipSelect(&chip_select, &HIGH_ADD);
	COMMAND = (chip_select | WRITE);					        // Chip select + WRITE
	BYTE_NUMBER = bcount;											        // Data to be writen
	I2C_BUFF = ptr;
	i2cSendStart(); 														      // STA = 1;		
	while (SM_BUSY);
}
#endif
// SMBus write function-----------------------------------------------------
// Writes a multiply byte at the specified memory location.
//
// *ptr = data byte to be written
// bcount = data byte count to write
// byte_address(8 bit) = memory location to be written into 
// chip_select = device address of EEPROM chip to be written to

#if defined(DEVICE_8)
void SM_Send8(BYTE chip_select, unsigned char byte_address,
	unsigned char* ptr, unsigned int bcount)
{
	while (SM_BUSY && inb(TWCR)&BV(TWSTO));					  // Wait for SMBus to be free.
	SM_BUSY = 1;						                          // Occupy SMBus (set to busy)
	// SMBus enabled, ACK on acknowledge cycle
	ADDR16 = 0; 																			// 8 bit adressing mode  
	HIGH_ADD = byte_address;													// Upper 8 address bits
	COMMAND = (chip_select | WRITE);			            // Chip select + WRITE
	BYTE_NUMBER = bcount;														  // Data to be writen
	I2C_BUFF = ptr;
	i2cSendStart(); 																	// STA = 1;	
	while (SM_BUSY);
}
#endif

// SMBus read function------------------------------------------------------
// Reads multiply byte from the specified memory location.
//
// *ptr = data byte from be read
// bcount = data byte count to read
// byte_address = memory address(16 bit) of byte to read
// chip_select = device address of EEPROM to be read from
#if defined(DEVICE_16)
void SM_Receive16(BYTE chip_select, unsigned int byte_address,
	unsigned char* ptr, unsigned int bcount)
{
	while (SM_BUSY && inb(TWCR)&BV(TWSTO)); 				  // Wait for bus to be free.	
	SM_BUSY = 1;					                            // Occupy SMBus (set to busy)
	// SMBus enabled, ACK on acknowledge cycle
	ADDR16 = 1; 																		  // 16 bit adressing mode		
	ADDR16_SENT = 0;
	HIGH_ADD = ((byte_address >> 8) & 0x00FF);			  // Upper 8 address bits
	LOW_ADD = (byte_address & 0x00FF);							  // Lower 8 address bits
	MakeChipSelect(&chip_select, &HIGH_ADD);
	COMMAND = (chip_select | READ); 								  // Chip select + READ
	BYTE_NUMBER = bcount; 													  // Byte count to read 
	I2C_BUFF = ptr;	
	i2cSendStart(); 																  // STA = 1;
	while (SM_BUSY);																  // Wait for transfer to finish
}
#endif
// SMBus random read function------------------------------------------------------
// Reads BYTE_NUMBER byte from the specified memory location.
//
// *ptr = data byte from be read
// bcount = data byte count to read
// byte_address = memory address(8 bit) of byte to read 
// chip_select = device address of EEPROM to be read from
#if defined(DEVICE_8)
void SM_Receive8(BYTE chip_select, unsigned char byte_address,
	unsigned char* ptr, unsigned int bcount)
{
	while (SM_BUSY && inb(TWCR)&BV(TWSTO));					  // Wait for bus to be free.
	SM_BUSY = 1;					                            // Occupy SMBus (set to busy)
	// SMBus enabled, ACK on acknowledge cycle
	ADDR16 = 0; 																		  // 8 bit adressing mode 	 
	HIGH_ADD = byte_address;												  // Upper 8 address bits
	COMMAND = (chip_select | READ);									  // Chip select + READ
	BYTE_NUMBER = bcount;				                      // Byte count to read 
	I2C_BUFF = ptr;	
	i2cSendStart();					                          // Start transfer
	while (SM_BUSY);				                          // Wait for transfer to finish
}
#endif

//! I2C (TWI) interrupt service routine

#pragma vector=TWSI_vect
__interrupt void TWI_interrupt(void)
{
	// read status bits
	BYTE status = inb(TWSR) & TWSR_STATUS_MASK;
	BYTE action = 0;                                  // STO,STA,EA bit temp store reg 
	switch (status)
	{
		// Master General
	case TW_START:
		// 0x08: Sent start condition
		outb(TWDR, COMMAND & 0xFE);			                // Load address of the slave to be accessed.
		REP_START = 0;
		I2C_COUNT = 0;
		//STA = 0;	
		break;
	case TW_REP_START:
		// 0x10: Sent repeated start condition
		//		printf("I2C: M->START\r\n");		
		// send device address
		outb(TWDR, COMMAND);				                    // COMMAND should hold slave address + R.
		I2C_COUNT = 0;
		//STA = 0;													
		break;

		// Master Transmitter & Receiver status codes
	case TW_MT_SLA_ACK:
		outb(TWDR, HIGH_ADD);			
		break;
	case TW_MT_DATA_ACK:
		// 0x28: Data acknowledged
		//		printf("I2C: MT->SLA_ACK or DATA_ACK\r\n");
		// Master Transmitter: Data byte transmitted.  ACK received.
		// This state is used in both READ and WRITE operations.	Check BYTE_NUMBER
		// for memory address status - if only HIGH_ADD has been sent, load LOW_ADD.
		// If LOW_ADD has been sent, check COMMAND for R/W value to determine next state.

		if (ADDR16 && !ADDR16_SENT) 				 // If adress is 16 bit and not sent
		{
			outb(TWDR, LOW_ADD);								// has been sent.
			ADDR16_SENT = 1;									// Sent flagged
		}
		else
		{
			if ((COMMAND & 0x01) && !REP_START) // If R/W=READ, sent repeated START.
			{
				REP_START = 1;									// REP_START	flagged
				action = BV(TWSTA); 								// Activate repeated start		
			}
			else
			{
				if (I2C_COUNT >= BYTE_NUMBER) 		 // If LAST_BYTE, transfer is finished.
				{
					action = BV(TWSTO);
				}
				else
				{
					outb(TWDR, I2C_BUFF[I2C_COUNT++]);// If R/W=WRITE, load byte to write.										
				}
			}
		}		
		break;
	case TW_MR_DATA_NACK:
		// 0x58: Data received, NACK reply issued
		//		printf("I2C: MR->DATA_NACK\r\n");
		// store final received data byte
		I2C_BUFF[I2C_COUNT] = inb(TWDR);
		action = BV(TWSTO);
		break;
		// continue to transmit STOP condition
	case TW_MR_SLA_NACK:
		// 0x48: Slave address not acknowledged
	case TW_MT_SLA_NACK:
		// 0x20: Slave address not acknowledged
	case TW_MT_DATA_NACK:
		// 0x30: Data not acknowledged
		//		printf("I2C: MTR->SLA_NACK or MT->DATA_NACK\r\n");		
		// transmit stop condition, enable SLA ACK
		action = BV(TWSTO);
		// set state		
		break;
	case TW_MT_ARB_LOST:
		// 0x38: Bus arbitration lost
		//case TW_MR_ARB_LOST:				// 0x38: Bus arbitration lost
		//		printf("I2C: MT->ARB_LOST\r\n");		
		// release bus
		//outb(TWCR, (inb(TWCR)&TWCR_CMD_MASK)|BV(TWINT));
		// set state
		SM_BUSY = 0;
		// release bus and transmit start when bus is free
		//outb(TWCR, (inb(TWCR)&TWCR_CMD_MASK)|BV(TWINT)|BV(TWSTA));
		break;
	case TW_MR_DATA_ACK:
		// 0x50: Data acknowledged
		//		printf("I2C: MR->DATA_ACK\r\n");		
		// store received data byte
		//	PORTB = PORTB ^ (1<<PORTB5);
		I2C_BUFF[I2C_COUNT++] = inb(TWDR);								
		if (I2C_COUNT < BYTE_NUMBER - 1)
		{
			action = BV(TWEA);
		}
		break; 		
		// fall-through to see if more bytes will be received
	case TW_MR_SLA_ACK:
		// 0x40: Slave address acknowledged
		//		printf("I2C: MR->SLA_ACK\r\n");		
		if (BYTE_NUMBER != 1)
		{
			action = BV(TWEA);										// NACK sent on acknowledge cycle.		
		}  
		break;

		// Slave Receiver status codes
	case TW_SR_SLA_ACK:
		// 0x60: own SLA+W has been received, ACK has been returned
	case TW_SR_ARB_LOST_SLA_ACK:
		// 0x68: own SLA+W has been received, ACK has been returned
	case TW_SR_GCALL_ACK:
		// 0x70:		 GCA+W has been received, ACK has been returned
	case TW_SR_ARB_LOST_GCALL_ACK:
		// 0x78:		 GCA+W has been received, ACK has been returned
		//		printf("I2C: SR->SLA_ACK\r\n");		
		// we are being addressed as slave for writing (data will be received from master)
		// set state
		I2cState = I2C_SLAVE_RX;
		// prepare buffer
		//I2cReceiveDataIndex = 0;
		// receive data byte and return ACK
		outb(TWCR, (inb(TWCR) & TWCR_CMD_MASK) | BV(TWEA));
		break;
	case TW_SR_DATA_ACK:
		// 0x80: data byte has been received, ACK has been returned
	case TW_SR_GCALL_DATA_ACK:
		// 0x90: data byte has been received, ACK has been returned
		//		printf("I2C: SR->DATA_ACK\r\n");		
		// get previously received data byte
		//		I2cReceiveData[I2cReceiveDataIndex++] = inb(TWDR);
		// check receive buffer status
		//	if(I2cReceiveDataIndex < I2C_RECEIVE_DATA_BUFFER_SIZE)
		//{
		// receive data byte and return ACK
		//	i2cReceiveByte(TRUE);
		//outb(TWCR, (inb(TWCR)&TWCR_CMD_MASK)|BV(TWINT)|BV(TWEA));
		//}
		//else
		//{
		// receive data byte and return NACK
		//	i2cReceiveByte(FALSE);
		//outb(TWCR, (inb(TWCR)&TWCR_CMD_MASK)|BV(TWINT));
		//}
		break;
	case TW_SR_DATA_NACK:
		// 0x88: data byte has been received, NACK has been returned
	case TW_SR_GCALL_DATA_NACK:
		// 0x98: data byte has been received, NACK has been returned
		//		printf("I2C: SR->DATA_NACK\r\n");	
		// receive data byte and return NACK
		//i2cReceiveByte(FALSE);
		//outb(TWCR, (inb(TWCR)&TWCR_CMD_MASK)|BV(TWINT));
		break;
	case TW_SR_STOP:
		// 0xA0: STOP or REPEATED START has been received while addressed as slave
		//		printf("I2C: SR->SR_STOP\r\n");
		// switch to SR mode with SLA ACK
		outb(TWCR, (inb(TWCR) & TWCR_CMD_MASK) | BV(TWEA));
		// i2c receive is complete, call i2cSlaveReceive
		//if(i2cSlaveReceive) i2cSlaveReceive(I2cReceiveDataIndex, I2cReceiveData);
		// set state
		SM_BUSY = 0;	
		break;

		// Slave Transmitter
	case TW_ST_SLA_ACK:
		// 0xA8: own SLA+R has been received, ACK has been returned
	case TW_ST_ARB_LOST_SLA_ACK:
		// 0xB0:		 GCA+R has been received, ACK has been returned
		//		printf("I2C: ST->SLA_ACK\r\n");		
		// we are being addressed as slave for reading (data must be transmitted back to master)
		// set state
		I2cState = I2C_SLAVE_TX;
		// request data from application
		//	if(i2cSlaveTransmit) I2cSendDataLength = i2cSlaveTransmit(I2C_SEND_DATA_BUFFER_SIZE, I2cSendData);
		// reset data index
		//	I2cSendDataIndex = 0;
		// fall-through to transmit first data byte
	case TW_ST_DATA_ACK:
		// 0xB8: data byte has been transmitted, ACK has been received
		//		printf("I2C: ST->DATA_ACK\r\n");
		// transmit data byte
		//	outb(TWDR, I2cSendData[I2cSendDataIndex++]);
		//if(I2cSendDataIndex < I2cSendDataLength)
		// expect ACK to data byte
		//outb(TWCR, (inb(TWCR)&TWCR_CMD_MASK)|BV(TWEA));
		//else;
		// expect NACK to data byte
		//outb(TWCR, (inb(TWCR)&TWCR_CMD_MASK));
		break;
	case TW_ST_DATA_NACK:
		// 0xC0: data byte has been transmitted, NACK has been received
	case TW_ST_LAST_DATA:
		// 0xC8:
		//		printf("I2C: ST->DATA_NACK or LAST_DATA\r\n");
		// all done
		// switch to open slave
		outb(TWCR, (inb(TWCR) & TWCR_CMD_MASK) | BV(TWEA));
		// set state
		I2cState = I2C_IDLE;
		break;

		// Misc
	case TW_NO_INFO:
		// 0xF8: No relevant state information
		// do nothing
		//		printf("I2C: NO_INFO\r\n");
		break;
	case TW_BUS_ERROR:
		// 0x00: Bus error due to illegal start or stop condition
		//		printf("I2C: BUS_ERROR\r\n");
		// reset internal hardware and release bus
		outb(TWCR, (inb(TWCR) & TWCR_CMD_MASK) | BV(TWSTO) | BV(TWEA));
		// set state
		SM_BUSY = 0;
		break;
	}
	outb(TWCR, (inb(TWCR) & TWCR_CMD_MASK) | BV(TWINT) | action);
	if (action & BV(TWSTO))
	{
		SM_BUSY = 0;
	} 				
}

eI2cStateType i2cGetState(void)
{
	return I2cState;
}
