/*! \file ata.c \brief IDE-ATA hard disk interface driver. */
//*****************************************************************************
//
// File Name	: 'ata.c'
// Title		: IDE-ATA interface driver for hard disks
// Author		: Pascal Stang
// Date			: 11/22/2000
// Revised		: 4/19/2003
// Version		: 0.3
// Target MCU	: Atmel AVR Series
// Editor Tabs	: 4
//
// NOTE: This code is currently below version 1.0, and therefore is considered
// to be lacking in some functionality or documentation, or may not be fully
// tested.  Nonetheless, you can expect most functions to work.
//
// This code is distributed under the GNU Public License
//		which can be found at http://www.gnu.org/licenses/gpl.txt
//
//*****************************************************************************

#include <pgmspace.h>
#include <stdio.h>
#include "config.h"
#include "types.h"
#include "defs.h"
#include "timer.h"
#include "vga.h"
#include "ata.h"

#define DEBUG_ATA	1

// global variables

// drive information
typeDriveInfo ataDriveInfo;
__flash const char  HexChars[] = "0123456789ABCDEF";

BYTE buffer[512];

//*****************************************************************************
void rprintfu04(unsigned char data)
{
	// print 4-bit hex value
//	char Character = data&0x0f;
//	if (Character>9)
//		Character+='A'-10;
//	else
//		Character+='0';
  putchar(HexChars[(data&0x0f)]);
}
//*****************************************************************************
// *** rprintfu08 ***
// prints an unsigned 8-bit number in hex (2 digits)
void rprintfu08(unsigned char data)
{
	// print 8-bit hex value
	rprintfu04(data>>4);
	rprintfu04(data);
}
//*****************************************************************************
// *** rprintfu16 ***
// prints an unsigned 16-bit number in hex (4 digits)
void rprintfu16(unsigned short data)
{
	// print 16-bit hex value
	rprintfu08(data>>8);
	rprintfu08(data);
}
//*****************************************************************************
// *** rprintfu32 ***
// prints an unsigned 32-bit number in hex (8 digits)
void rprintfu32(unsigned long data)
{
	// print 32-bit hex value
	rprintfu16(data>>16);
	rprintfu16(data);
}
//*****************************************************************************
void ataInit(void)
{

}
//*****************************************************************************
void ataDriveInit(BYTE *buffer)
{
	BYTE i;

	// read drive identity
	myprintf_PSTR("\r\nScanning IDE interface...\r\n");
	// Wait for drive to be ready
	//ataStatusWait(ATA_SR_BSY, ATA_SR_BSY);
  ataWaitBusy();
	// issue identify command
	ataWriteByte(ATA_REG_CMDSTATUS1, 0xEC);
	// wait for drive to request data transfer
	ataStatusWait(ATA_SR_DRQ, ATA_SR_DRQ);
	DelayUs(200);
	// read in the data
	ataReadDataBuffer((WORD*)buffer, 512);

	// set local drive info parameters
	ataDriveInfo.cylinders =		*( ((unsigned int*) buffer) + ATA_IDENT_CYLINDERS );
	ataDriveInfo.heads =			*( ((unsigned int*) buffer) + ATA_IDENT_HEADS );
	ataDriveInfo.sectors =			*( ((unsigned int*) buffer) + ATA_IDENT_SECTORS );
	ataDriveInfo.LBAsupport =		*( ((unsigned int*) buffer) + ATA_IDENT_FIELDVALID );
	ataDriveInfo.sizeinsectors =	*( (unsigned long*) (buffer + ATA_IDENT_LBASECTORS*2) );
	// copy model string
	for(i=0; i<40; i+=2)
	{
		// correct for byte order
		ataDriveInfo.model[i  ] = buffer[(ATA_IDENT_MODEL*2) + i + 1];
		ataDriveInfo.model[i+1] = buffer[(ATA_IDENT_MODEL*2) + i    ];
	}
	// terminate string
	ataDriveInfo.model[40] = 0;

	// process and print info
	if(ataDriveInfo.LBAsupport)
	{
		// LBA support
	  printf_P("Drive 0: %dMB ", ataDriveInfo.sizeinsectors/(1000000/512) );
		printf_P("LBA mode -- MODEL: ");
	}
	else
	{
		// CHS, no LBA support
		// calculate drive size
		ataDriveInfo.sizeinsectors = (unsigned long) ataDriveInfo.cylinders*
												ataDriveInfo.heads*ataDriveInfo.sectors;
		printf_P("Drive 0: %dMB ", ataDriveInfo.sizeinsectors/(1000000/512) );
		printf_P("CHS mode C=%d H=%d S=%d -- MODEL: ", ataDriveInfo.cylinders, ataDriveInfo.heads, ataDriveInfo.sectors );
	}
	// print model information	
	printf(ataDriveInfo.model); printf_P("\r\n");

	// initialize local disk parameters
	//ataDriveInfo.cylinders = ATA_DISKPARM_CLYS;
	//ataDriveInfo.heads = ATA_DISKPARM_HEADS;
	//ataDriveInfo.sectors = ATA_DISKPARM_SECTORS;

}
//*****************************************************************************
void ataDiskErr(void)
{
	unsigned char b;

	b = ataReadByte(ATA_REG_ERROR);	
	printf_P("ATA Error: ");
	rprintfu08(b);
	printf_P("\r\n");
}
//*****************************************************************************
void ataSetDrivePowerMode(BYTE DriveNo, BYTE mode, BYTE timeout)
{
	// select drive
	ataDriveSelect(DriveNo);
	// Wait for drive to be ready
	//ataStatusWait(ATA_SR_BSY, ATA_SR_BSY);
  ataWaitBusy();
	// set mode
	switch(mode)
	{
	case ATA_DISKMODE_SPINDOWN:		ataWriteByte(ATA_REG_CMDSTATUS1, ATA_CMD_SPINDOWN); break;
	case ATA_DISKMODE_SPINUP:		ataWriteByte(ATA_REG_CMDSTATUS1, ATA_CMD_SPINUP); break;
	case ATA_DISKMODE_SETTIMEOUT:
		ataWriteByte(ATA_REG_SECCOUNT, timeout);
		ataWriteByte(ATA_REG_CMDSTATUS1, ATA_CMD_IDLE_5SU);
		break;
	case ATA_DISKMODE_SLEEP:		ataWriteByte(ATA_REG_CMDSTATUS1, ATA_CMD_SLEEP); break;
	default:
		break;
	}
}
//*****************************************************************************
void ataPrintSector( BYTE *Buffer)
{
	BYTE i;
	WORD j;
	BYTE *buf;
	BYTE s;

	buf = Buffer;
	
	// print the low order address indicies
  printf_P("     00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F  0123456789ABCDEF\r\n");
  printf_P("     -----------------------------------------------  ---- ASCII -----\r\n");
	
	// print the data
	for(j=0; j<0x20; j++)
	{
		// print the high order address index for this line
		rprintfu16(j<<4);
		printf_P(" ");

		// print the hex data
		for(i=0; i<0x10; i++)
		{
			rprintfu08(buf[(j<<4)+i]);
			printf_P(" ");
		}
		
		// leave some space
		printf_P(" ");

		// print the ascii data
		for(i=0; i<0x10; i++)
		{
			s = buf[(j<<4)+i];
			// make sure character is printable
			if(s >= 0x20)
			{
				putchar(s);
			}
			else
			{
				putchar(0x20);
			}

		}
    printf_P("\r\n");
	}
}
//*****************************************************************************
void ataReadDataBuffer(WORD *Buffer, WORD numBytes)
{
	unsigned int i;

	//sbi(MCUCR, SRW);			// enable RAM waitstate

	// read data from drive
	for (i=0; i<(numBytes/16); i++)
	{
		// optimize by reading 16 bytes in-line before looping
		*Buffer++ = IoPortInW(ATA_REG_BASE + ATA_REG_DATAL);
		*Buffer++ = IoPortInW(ATA_REG_BASE + ATA_REG_DATAL);
		*Buffer++ = IoPortInW(ATA_REG_BASE + ATA_REG_DATAL);
		*Buffer++ = IoPortInW(ATA_REG_BASE + ATA_REG_DATAL);
		*Buffer++ = IoPortInW(ATA_REG_BASE + ATA_REG_DATAL);
		*Buffer++ = IoPortInW(ATA_REG_BASE + ATA_REG_DATAL);
		*Buffer++ = IoPortInW(ATA_REG_BASE + ATA_REG_DATAL);
		*Buffer++ = IoPortInW(ATA_REG_BASE + ATA_REG_DATAL);
	}
	//cbi(MCUCR, SRW);			// disable RAM waitstate
	
}
//*****************************************************************************
void ataWriteDataBuffer(WORD *Buffer, WORD numBytes)
{
	unsigned int i;

	//sbi(MCUCR, SRW);			// enable RAM waitstate

	// write data to drive
	for (i=0; i<(numBytes/16); i++) 	
	{
		// optimize by writing 16 bytes in-line before looping
		// keep byte order correct by using temp register
//		temp = *Buffer++;
//		*((volatile unsigned char*) ATA_REG_BASE + ATA_REG_DATAH) = *Buffer++;
//		*((volatile unsigned char*) ATA_REG_BASE + ATA_REG_DATAL) = temp;

    IoPortOutW(ATA_REG_BASE + ATA_REG_DATAL,*Buffer++);
    IoPortOutW(ATA_REG_BASE + ATA_REG_DATAL,*Buffer++);
    IoPortOutW(ATA_REG_BASE + ATA_REG_DATAL,*Buffer++);
    IoPortOutW(ATA_REG_BASE + ATA_REG_DATAL,*Buffer++);
    IoPortOutW(ATA_REG_BASE + ATA_REG_DATAL,*Buffer++);
    IoPortOutW(ATA_REG_BASE + ATA_REG_DATAL,*Buffer++);
    IoPortOutW(ATA_REG_BASE + ATA_REG_DATAL,*Buffer++);
    IoPortOutW(ATA_REG_BASE + ATA_REG_DATAL,*Buffer++);

	}
}
//*****************************************************************************
BYTE ataWaitBusy(void)
{
  BYTE rv;

  while ((rv=ataReadByte(ATA_REG_CMDSTATUS1))& ATA_SR_BSY);
  return(rv);
}
//*****************************************************************************
BYTE ataStatusWait(BYTE mask, BYTE waitStatus)
{
	register BYTE status;

	DelayUs(100);

	// wait for desired status
	while( ((status = ataReadByte(ATA_REG_CMDSTATUS1)) & mask) != waitStatus );

	return status;
}
//*****************************************************************************
unsigned char ataReadSectorsCHS(	unsigned char Drive,
											unsigned char Head,
											unsigned int Track,
											unsigned char Sector,
											unsigned int numsectors,
											unsigned char *Buffer)
{
  	unsigned char temp;

	// Wait for drive to be ready
//	temp = ataStatusWait(ATA_SR_BSY, ATA_SR_BSY);//
  temp = ataWaitBusy();

  	// Prepare parameters...
	ataWriteByte(ATA_REG_HDDEVSEL, 0xA0+(Drive ? 0x10:00)+Head); // CHS mode/Drive/Head
	ataWriteByte(ATA_REG_CYLHI, Track>>8);  		// MSB of track
	ataWriteByte(ATA_REG_CYLLO, Track);     		// LSB of track
  	ataWriteByte(ATA_REG_STARTSEC, Sector);    	// sector
	ataWriteByte(ATA_REG_SECCOUNT, numsectors);	// # of sectors

	// Issue read sector command...
  	ataWriteByte(ATA_REG_CMDSTATUS1, 0x21);

	// Wait for drive to be ready
//	temp = ataStatusWait(ATA_SR_BSY, ATA_SR_BSY);
  temp = ataWaitBusy();

	if (temp & ATA_SR_ERR)
	{
		printf_P("RD ERR\r\n");
		return 1;
	}

	// Wait for drive to request data transfer
	ataStatusWait(ATA_SR_DRQ,ATA_SR_DRQ);

	// read data from drive
	ataReadDataBuffer((WORD*)Buffer, 512*numsectors);

	// Return the error bit from the status register...
	temp = ataReadByte(ATA_REG_CMDSTATUS1);	// read status register

	return (temp & ATA_SR_ERR) ? 1:0;
}
//*****************************************************************************
unsigned char ataWriteSectorsCHS(unsigned char Drive,
											unsigned char Head,
											unsigned int Track,
											unsigned char Sector,
											unsigned int numsectors,
											unsigned char *Buffer)
{
  	unsigned char temp;

	// Wait for drive to be ready
//	temp = ataStatusWait(ATA_SR_BSY, ATA_SR_BSY);
  temp = ataWaitBusy();

  	// Prepare parameters...
	ataWriteByte(ATA_REG_HDDEVSEL, 0xA0+(Drive ? 0x10:00)+Head); // CHS mode/Drive/Head
	ataWriteByte(ATA_REG_CYLHI, Track>>8);  		// MSB of track
	ataWriteByte(ATA_REG_CYLLO, Track);     		// LSB of track
  	ataWriteByte(ATA_REG_STARTSEC, Sector);    	// sector
	ataWriteByte(ATA_REG_SECCOUNT, numsectors);	// # of sectors

	// Issue write sector command
  	ataWriteByte(ATA_REG_CMDSTATUS1, 0x31);

	//delay(100);

	// Wait for drive to request data transfer
	ataStatusWait(ATA_SR_DRQ,ATA_SR_DRQ);

	// write data to drive
	ataWriteDataBuffer((WORD*)Buffer, 512*numsectors);
	
	// Wait for drive to finish write
//	temp = ataStatusWait(ATA_SR_BSY, ATA_SR_BSY);
  temp = ataWaitBusy();

	// check for errors
	if (temp & ATA_SR_ERR)
	{
		printf_P("WR ERR\r\n");
		return 1;
	}

	// Return the error bit from the status register...
	return (temp & ATA_SR_ERR) ? 1:0;
}
//*****************************************************************************
unsigned char ataReadSectorsLBA(	unsigned char Drive,
			unsigned long lba, unsigned int numsectors,unsigned char *Buffer)
{
  	unsigned int cyl, head, sect;
  	unsigned char temp;

#ifdef DEBUG_ATA
	printf_P("ATA LBA read ");
	rprintfu32(lba); printf_P(" ");
	rprintfu16(numsectors); printf_P(" ");
	rprintfu16((unsigned int)Buffer);
	printf_P("\r\n");
#endif

	sect = (int) ( lba & 0x000000ffL );
	lba = lba >> 8;
	cyl = (int) ( lba & 0x0000ffff );
	lba = lba >> 16;
	head = ( (int) ( lba & 0x0fL ) ) | ATA_HEAD_USE_LBA;

	temp = ataReadSectorsCHS( Drive, head, cyl, sect, numsectors, Buffer );

	if(temp)
		ataDiskErr();
	return temp;
}
//*****************************************************************************
unsigned char ataWriteSectorsLBA(	unsigned char Drive,
												unsigned long lba,
												unsigned int numsectors,
                            			unsigned char *Buffer)
{
	unsigned int cyl, head, sect;
	unsigned char temp;

#ifdef DEBUG_ATA
	printf_P("ATA LBA write ");
	rprintfu32(lba); printf_P(" ");
	rprintfu16(numsectors); printf_P(" ");
	rprintfu16((unsigned int)Buffer);
	printf_P("\r\n");
#endif

	sect = (int) ( lba & 0x000000ffL );
	lba = lba >> 8;
	cyl = (int) ( lba & 0x0000ffff );
	lba = lba >> 16;
	head = ( (int) ( lba & 0x0fL ) ) | ATA_HEAD_USE_LBA;

	temp = ataWriteSectorsCHS( Drive, head, cyl, sect, numsectors, Buffer );

	if(temp)
		ataDiskErr();
	return temp;
}                            		

//*****************************************************************************
unsigned char ataReadSectors(	unsigned char Drive,
										unsigned long lba,
										unsigned int numsectors,
                            	unsigned char *Buffer)
{
  	unsigned int cyl, head, sect;
  	unsigned char temp;

	// check if drive supports native LBA mode
	if(ataDriveInfo.LBAsupport)
	{
		// drive supports using native LBA
		temp = ataReadSectorsLBA(Drive, lba, numsectors, Buffer);
	}
	else
	{
		// drive required CHS access
		#ifdef DEBUG_ATA
			// do this defore destroying lba
			printf_P("ATA LBA for CHS read: ");
			printf_P("LBA="); rprintfu32(lba); printf_P(" ");
		#endif

		// convert LBA to pseudo CHS
		// remember to offset the sector count by one
		sect = (BYTE) (lba % ataDriveInfo.sectors)+1;
		lba = lba / ataDriveInfo.sectors;
		head = (BYTE) (lba % ataDriveInfo.heads);
		lba = lba / ataDriveInfo.heads;
		cyl = (WORD) lba;

		#ifdef DEBUG_ATA
			printf_P("C:H:S=");
			rprintfu16(cyl); printf_P(":");
			rprintfu08(head); printf_P(":");
			rprintfu08(sect); printf_P("\r\n");
		#endif

		temp = ataReadSectorsCHS( Drive, head, cyl, sect, numsectors, Buffer );
	}

	if(temp)
		ataDiskErr();
	return temp;
}
//*****************************************************************************
unsigned char ataWriteSectors(unsigned char Drive,
										unsigned long lba,
										unsigned int numsectors,
                            	unsigned char *Buffer)
{
  	unsigned int cyl, head, sect;
  	unsigned char temp;

	// check if drive supports native LBA mode
	if(ataDriveInfo.LBAsupport)
	{
		// drive supports using native LBA
		temp = ataWriteSectorsLBA(Drive, lba, numsectors, Buffer);
	}
	else
	{
		// drive required CHS access
		#ifdef DEBUG_ATA
			// do this defore destroying lba
			printf_P("ATA LBA for CHS write: ");
			printf_P("LBA="); rprintfu32(lba); printf_P(" ");
		#endif

		// convert LBA to pseudo CHS
		// remember to offset the sector count by one
		sect = (BYTE) (lba % ataDriveInfo.sectors)+1;
		lba = lba / ataDriveInfo.sectors;
		head = (BYTE) (lba % ataDriveInfo.heads);
		lba = lba / ataDriveInfo.heads;
		cyl = (WORD) lba;

		#ifdef DEBUG_ATA
			printf_P("C:H:S=");
			rprintfu16(cyl);  printf_P(":");
			rprintfu08(head); printf_P(":");
			rprintfu08(sect); printf_P("\r\n");
		#endif

		temp = ataWriteSectorsCHS( Drive, head, cyl, sect, numsectors, Buffer );
	}

	if(temp)
		ataDiskErr();
	return temp;
}                            		
//*****************************************************************************
void ataDriveSelect(BYTE DriveNo)
{
	ataWriteByte(ATA_REG_HDDEVSEL, 0xA0+(DriveNo ? 0x10:00)); // Drive selection
}

//----------------------------------------------------------------------------
// Set drive mode (STANDBY, IDLE)
//----------------------------------------------------------------------------
/*#define STANDBY 0
#define IDLE    1
#define SLEEP   2
*/

/*
unsigned char SetMode(unsigned char DriveNo, unsigned char Mode, unsigned char PwrDown)
{
  WriteBYTE(CMD, 6, 0xA0 + (DriveNo ? 0x10:0x00)); // Select drive
  WriteBYTE(CMD, 2, (PwrDown ? 0x01:0x00)); // Enable automatic power down
  switch (Mode)
  {
    case STANDBY: WriteBYTE(CMD,7, 0xE2); break;
    case IDLE:    WriteBYTE(CMD,7, 0xE3); break;
    // NOTE: To recover from sleep, either issue a soft or hardware reset !
    // (But not on all drives, f.ex seagate ST3655A it's not nessecary to reset
    // but only to go in Idle mode, But on a Conner CFA170A it's nessecary with
    // a reset)
    case SLEEP:   WriteBYTE(CMD,7, 0xE6); break;
  }
  Timer10mSec=10000;
  while ((ReadBYTE(CMD,7) & 0xC0)!=0x40 && Timer10mSec); // Wait for DRDY & NOT BUSY
  if (Timer10mSec==0) return 0xFF;                       //   or timeout

  // Return the error register...
  return ReadBYTE(CMD, 1);
}

*/
//*****************************************************************************
BYTE ataReadByte(BYTE reg)
{
	return(IoPortInB(ATA_REG_BASE + reg));
}
//*****************************************************************************
void ataWriteByte(BYTE reg, BYTE data)
{	
	IoPortOutB(ATA_REG_BASE + reg,data);
}
//*****************************************************************************
void ataShowRegisters(unsigned char DriveNo)
{
	ataWriteByte(ATA_REG_HDDEVSEL, 0xA0 + (DriveNo ? 0x10:0x00)); // Select drive
	
	printf_P("R0: DATALOW  = 0x");	rprintfu08(ataReadByte(ATA_REG_DATAL	));		printf_P(" \r\n");
	printf_P("R1: ERROR    = 0x");	rprintfu08(ataReadByte(ATA_REG_ERROR	));		printf_P(" \r\n");
	printf_P("R2: SECT CNT = 0x");	rprintfu08(ataReadByte(ATA_REG_SECCOUNT));	printf_P(" \r\n");
	printf_P("R3: SECT NUM = 0x");	rprintfu08(ataReadByte(ATA_REG_STARTSEC));	printf_P(" \r\n");
	printf_P("R4: CYL LOW  = 0x");	rprintfu08(ataReadByte(ATA_REG_CYLLO	));		printf_P(" \r\n");
	printf_P("R5: CYL HIGH = 0x");	rprintfu08(ataReadByte(ATA_REG_CYLHI	));		printf_P(" \r\n");
	printf_P("R6: HEAD/DEV = 0x");	rprintfu08(ataReadByte(ATA_REG_HDDEVSEL));	printf_P(" \r\n");
	printf_P("R7: CMD/STA  = 0x");	rprintfu08(ataReadByte(ATA_REG_CMDSTATUS1));printf_P("\r\n");
}
//*****************************************************************************
unsigned char ataSWReset(void)
{
	ataWriteByte(ATA_REG_HDDEVSEL, 0x06);	// SRST and nIEN bits
	DelayUs(10);	// 10uS delay
	ataWriteByte(ATA_REG_HDDEVSEL, 0x02);	// nIEN bits
	DelayUs(10);	// 10 uS delay

   while( (ataReadByte(ATA_REG_CMDSTATUS1) & 0xC0) != 0x40 ); // Wait for DRDY and not BSY

  	return ataReadByte(ATA_REG_CMDSTATUS1) + ataReadByte(ATA_REG_ERROR);
}
//*****************************************************************************
/*
unsigned char ATA_Idle(unsigned char Drive)
{

  ataWriteByte(CMD, 6, 0xA0 + (Drive ? 0x10:0x00)); // Select drive
  ataWriteByte(CMD,7, 0xE1);

  while ((ataReadByte(CMD,7) & 0xC0)!=0x40); // Wait for DRDY & NOT BUSY

  // Return the error register...
  return ataReadByte(CMD, 1);
}
*/
