#include <stdio.h>
#include <string.h>
#include <regat89c51.h>
#include <math.h>
#include "definitions.h"
#include "lcd.h"
#include "max6675.h"
#include "24lc64.h"
#include "keys.h"
#include "rs_232.h"

volatile unsigned int timer0count;
volatile bit timer0flag;
volatile bit secflag;
bit tmp_log;

char str_buf[20];

char menu_item,prev_menu_item;
unsigned char power_heat1,power_heat2;	// power_heatx stores the current powersetting for heatelement x
										// range (0 .. 50)
unsigned char powercount;				// used in int_handler
													
struct Tprofile profile;
unsigned char* p_prof;
unsigned char oven_overshoot;			// overshoot in oC at dT/dt = 1 oC/s


void secbeat(void);


void timer0handler (void) interrupt 1 using 2 
{
  if (timer0count-- == 0)
  {
    timer0flag = 1;
	timer0count = T0COUNTVAL;
  }
  if ((timer0count % (T0COUNTVAL/25))==0)
  {
  // 50 times a second (will be replaced by int. handler for INT0
  // Keypress handler
  //  LED2=~LED2;
	if (++powercount>49) {powercount=0; secflag=1;}
	if (power_heat1>powercount) {HEAT1=0;LED1=0;} else {HEAT1=1;LED1=1;}
	if (power_heat2>powercount) {HEAT2=0;LED2=0;} else {HEAT2=1;LED2=1;}
	if (powercount==0) LED3=~LED3;
  }
}

void init_controller (void)
{
// Initialize Timer 0  
  TMOD = 0x22; 					// timer0 and timer1 8bit reload
  TH0 = 0x06 ;  				// Timer 0 : 1 interrupt each 500uSec
  TL0 = 0x06; 					// 
  TR0 = 1;						// Start timer 0
// Initialize Timer 1 as baudgenerator for serial port
  TH1 = 0xF3;					// Timer 1 : 4800 Baud at 12MHz clock
  TL1 = 0xF3;
  TR1 = 1;						// Start timer 1;
// Configure UART
  PCON = 0x80;					// SMOD = 1 : Baud = timer1 overflow/16 (instead of /32)
  TCLK = 0;
  RCLK = 0;  
  SCON = 0X50;					// 8-BIT uart
// Initialize Timer 0 int handler
  timer0count = T0COUNTVAL;
  timer0flag  = 0;
// Misc
 // AUXR = AUXR | 0X19; 		// Disable ALE, RESET is input only, WDT disable in IDLE-mode
  
// Initialize interrupts
  ES = 1;						// Enable serialport interrupt
  ET1 = 0;						// Disable timer1 interrupt
  ET0 = 1;						// Enable timer0 interrupt
  EA = 1;						// Enable interrupts global
}

void wait_1ms (void)
{
  unsigned int end_point;
  bit EA_temp;
  EA_temp = EA;
  EA = 0;				// Disable interrupts loading end_point is a multi-instruction operation on a volatile
						// variable, therefore this must be done atomic
  end_point = timer0count;
  EA = EA_temp;			// Restore global interrupt enable
  if (end_point >=3 )   // Set end_point to 1,5mS later. This ensures that the delay is between 1mS and 1,5mS
    end_point = end_point -3; 
  else
    end_point = end_point + (T0COUNTVAL -3);
  while (timer0count != end_point) {}; 
}

void secbeat(void)
{
  while (secflag!=1){};
  secflag=0;
}

void calibrate(void)
{
  signed int tmp;
  unsigned char phase;
  signed int deltaT,deltaT100;
  lcd_print_line ("CAL : COOLING",0);
//  power_heat1=50;						// Switch on both heatelements
//  power_heat2=50;
  phase=0;								// assume we have to cool down first
  deltaT=0;								// Start from scratch
  while (phase<3)
  {
    tmp= read_temperature();
	if ((phase==0) && (tmp<(50*4))) 
	{
	  phase = 1; 
	  lcd_print_line ("CAL : HEATING",0); 
	  power_heat1=50; power_heat2=50;
	}
	if ((phase==1) && (tmp>(100*4))) 
	{
	  phase = 2; 
	  lcd_print_line ("CAL : OVERSHOOT",0); 
	  power_heat1=0; power_heat2=0;
	  deltaT100=deltaT;					// remember the deltaT at 100 C
	}
	if ((phase==2) && (deltaT<0))	
	{
	  phase =3; 
	  lcd_print_line (" CAL : DONE",0);
	  oven_overshoot = ((long) (((long)tmp * 25)-(10000))/deltaT100);
	  sprintf (str_buf,"overshoot:%bu%c",oven_overshoot,0xdf);
	  lcd_print_line (str_buf,1);
	  while ((keys_get() & KEY_ENTER)==0){};
      write_byte_24lc64 (0x00,oven_overshoot);
      write_byte_24lc64 (0x01,~oven_overshoot);
	}
	sprintf (str_buf,"T:%3.3u%cC d:%d.%2.2d%cC",(tmp/4),0xdf,deltaT/100,abs((deltaT % 100)),0xdf);
    lcd_print_line (str_buf,1);

    secbeat();
	deltaT = ((3*deltaT)+(((int)(read_temperature())-tmp)*25))/4;// read_temp() returns temperature * 4. *25 results in
														// degrees Celsius * 100;
  }
}

void control(void)
{
  unsigned char phase,keys;
  signed int tmp,des_tmp,des_slope,prev_tmp,deltaT,tempsoak;
  unsigned int timer;
  unsigned int EEPROM_adr;

  lcd_print_line ("PRE-COOLING",0);
  phase=0;
  EEPROM_adr = LOG_START_ADR;							// pointer to where the 
  while (phase<10)
  {
    tmp = read_temperature();
	switch (phase)
	{
	  case 0 : 
 	    des_tmp=50;
        if (tmp<50*4) 
		{
		  phase=1;
          des_tmp = profile.pre_endtemp;
		  des_slope = profile.pre_slope;
        }
		break;
	  case 1 : // PREHEAT
	    sprintf (str_buf,"PREHEAT %u%cC",des_tmp,0xdf);
		lcd_print_line (str_buf,0);
	    if ((tmp/4) >= des_tmp)
	    {
	      phase = 2;   // proceed to soak-phase;
		  timer = profile.soak_time;
		  lcd_print_line ("SOAK",0);
		  tempsoak = profile.soak_endtemp;
	    }
		break;
	  case 2 : // SOAK
        sprintf (str_buf,"%2.2u:%2.2u",timer/60,timer%60);	// Display remaining time
	    lcd_print_at (str_buf, 9,0);
		// Calculate the desired temperature. Sliding scale from pre_endtemp to soak_endtemp
		des_tmp = (((profile.soak_time-timer)*tempsoak)+
		           (timer*profile.pre_endtemp))/profile.soak_time;
		if (timer==0)
		{
		  phase = 3;	// proceed to dwell-phase;
		  des_tmp=profile.reflow_peaktemp;
		  lcd_print_line ("DWELL",0);
		}
	    break;
	  case 3 : // DWELL
	    if (tmp/4>(des_tmp-5))
		{
		  phase = 4;	// proceed to reflow-phase
		  lcd_print_line ("REFLOW",0);
		  timer = profile.reflow_time;
		}
		break;
	  case 4 : // REFLOW
        sprintf (str_buf,"%2.2u:%2.2u",timer/60,timer%60);	// Display remaining time
	    lcd_print_at (str_buf, 9,0);
		if (timer ==0)
		{
		  phase = 5;	// proceed to cooldown-phase
		  des_tmp = 0;
		  lcd_print_line ("COOLING",0);
		}
	  case 5 : // COOLDOWN
	    if ((tmp/4)<50) phase = 10;	// DONE!
      default:
        break;	        	    
     }      
	 deltaT = ((3*deltaT)+((tmp-prev_tmp)*25))/4;
     prev_tmp=tmp;
     tmp=tmp/4;											// convert tmp to oC

     if (tmp<des_tmp)
	 {
	   if (tmp+((oven_overshoot*deltaT)/100)<des_tmp) {power_heat1=50; power_heat2=50;}
	   else {power_heat1=0;power_heat2=0;}
	 }
	 else
	 {
	   if (tmp+((oven_overshoot*deltaT)/50)<des_tmp) {power_heat1=25; power_heat2=25;}
	   else {power_heat1=0;power_heat2=0;}
	 }

	 if ((phase==0) | (phase==5)) {power_heat1=0;power_heat2=0;}	// During cooldown the elements must be
	 																// inactive

     if (phase==2) 
	   sprintf (str_buf,"T%3.3d%cC end:%3.3d%cC",tmp,0xdf,tempsoak,0xdf);
	 else																   
	   sprintf (str_buf,"T%3.3d%cC to:%3.3d%cC",tmp,0xdf,des_tmp,0xdf);

	 lcd_print_line (str_buf,1);
     sprintf (str_buf,"%bu,%d,%d,%bd,%bd\n\r",phase,tmp,des_tmp,power_heat1,power_heat2);
	 printstr_rs232(str_buf);
     if (tmp_log)
	 {
       write_byte_24lc64 (EEPROM_adr,(tmp & 0x00ff));
       EEPROM_adr++;
       write_byte_24lc64 (EEPROM_adr,(tmp >>8));
       EEPROM_adr++;	   
	 }
	 while (secflag==0) 				// wait for end of second...
	 {
	   keys=keys_get();
       if (keys & KEY_EXIT)				// Stop the cycle
	   { power_heat1=0; power_heat2=0; phase=10; }
	   									// Adjust desired temperature. In the case of SOAK-phase
										// adjust the endtemperature, because the desired temperature
										// is calculated every second.
	   if (keys & KEY_UP) if (phase==2) tempsoak++; else des_tmp++;
	   if (keys & KEY_DOWN) if (phase==2) tempsoak--; else des_tmp--;
	 }
	 secflag = 0;
	 timer--;
   }
   if (tmp_log)
   {
     write_byte_24lc64 (EEPROM_adr,0);
     EEPROM_adr++;
     write_byte_24lc64 (EEPROM_adr,0);
   }
}

void edit (void)
{
  unsigned char prev_param,param,keys;
  unsigned int prev_value,value,min_value,max_value;
  bit done,show;
  param=1; prev_param=param;
  value = profile.pre_slope; max_value=5; min_value=1;
  done=0; show=1;
  lcd_print_line ("EDIT",0);
  while (done==0)
  {
    keys = keys_get();
	if (keys & KEY_RIGHT) {show=1; param++;}
	else
    if (keys & KEY_LEFT) {show=1; param--;}
    else
    if (keys & KEY_UP) {show=1; value++;}
    else if (keys & KEY_DOWN) {show=1; value--;}
    if (value<min_value) value=min_value;
    if (value>max_value) value=max_value;
    
    if (param<1) param=7;
    if (param>7) param=1;		
    if (prev_param!=param)		
	{
	  switch (param)
	  {
	    case 1: value = profile.pre_slope; max_value=5; min_value=1; break;
	    case 2: value = profile.pre_endtemp; max_value=200; min_value=75; break;
	    case 3: value = profile.soak_time; max_value=600; min_value=10;break;
	    case 4: value = profile.soak_endtemp; max_value=200; min_value=75; break;
	    case 5: value = profile.reflow_time; max_value=60; min_value=1; break;
	    case 6: value = profile.reflow_peaktemp; max_value=300; min_value=150; break;
	    default : break;
	  }
	}
	switch (param)
	{
	  case 1: profile.pre_slope=value; break;
	  case 2: profile.pre_endtemp=value; break;
	  case 3: profile.soak_time=value; break;
	  case 4: profile.soak_endtemp=value; break;
	  case 5: profile.reflow_time=value; break;
	  case 6: profile.reflow_peaktemp=value; break;
	  default : break;
    }
    if ((show==1)||(prev_param!=param)||(prev_value!=value))
    {
      switch (param)
   	  {
        case 1 : // PRE_SLOPE
	      lcd_print_line ("EDIT : preheat",0);
    	  sprintf (str_buf,"slope:%2u%cC/s",value,0xdf);
	      lcd_print_line (str_buf,1);
	      break;
	    case 2 : // PRE_ENDTEMP
	      lcd_print_line  ("EDIT : preheat",0);
	      sprintf (str_buf,"temp : %3u%cC",value,0xdf);
	      lcd_print_line (str_buf,1);
	      break;
	    case 3 : // SOAK_TIME
	      lcd_print_line  ("EDIT : soak",0);
	      sprintf (str_buf,"time  %2.2u:%2.2u",value/60,value%60);
	      lcd_print_line (str_buf,1);
	    break;
  	    case 4 : // SOAK_ENDTEMP
	      lcd_print_line  ("EDIT : soak",0);
	      sprintf (str_buf,"endtemp : %3u%cC",value,0xdf);
	      lcd_print_line (str_buf,1);
	      break;
	    case 5 : // REFLOW_TIME
	      lcd_print_line  ("EDIT : reflow",0);
	      sprintf (str_buf,"time : %2u sec",value);
	      lcd_print_line (str_buf,1);
	      break;
 	    case 6 : // REFLOW_ENDTEMP
	      lcd_print_line  ("EDIT : reflow",0);
	      sprintf (str_buf,"temp : %3u%cC",value,0xdf);
	      lcd_print_line (str_buf,1);
	      break;
	    case 7 : // OK
	      lcd_print_line ("EDIT  <END>",0);
		  lcd_print_line ("",1);
      }
	  prev_param=param;
    }
    if (keys& KEY_ENTER) done=1;
    prev_value=value;
  }
  lcd_print_line ("Save settings?",0);
  lcd_print_line ("[ENT] y [ESC] n",1);
  while (1)
  {
    keys=keys_get();
	p_prof =  (unsigned char*) &profile;
	if (keys & KEY_ENTER)
	{
	  for (param=0;param<sizeof(profile);param++)
	    write_byte_24lc64 (param+PROFILES_START_ADR, *(unsigned char*)p_prof++);		// stores EEPROM
	  return;
	}
	if (keys & KEY_EXIT)
	  return;
  }
}


void send_log(void)
{
  unsigned int adr,dat;
  lcd_print_line ("   Sending log",0);
  lcd_print_line ("   4800,8,1,N",1);
  adr = LOG_START_ADR;
  dat=1;
  while ((dat!=0)&(adr<LOG_END_ADR))
  {
    dat = read_byte_24lc64 (adr)+(read_byte_24lc64(adr+1)<<8);
    adr++;adr++;
	sprintf (str_buf,"%u\n\r",dat);
	printstr_rs232 (str_buf);
  }
}

void log(void)
{
  unsigned char keys,change;
  keys=keys_get(); change=1;
  lcd_print_line ("press [>] to send",1); 
  while ((keys & KEY_ENTER)==0)
  {
    keys=keys_get();
    if (keys & KEY_UP) {tmp_log=1;change=1;}
	if (keys & KEY_DOWN) {tmp_log=0;change=1;}
	if (keys & KEY_RIGHT) {send_log(); return;}
    if (change)
	{
	  change=0;
      if (tmp_log)
        lcd_print_line ("LOGGING : ON",1);
	  else
	    lcd_print_line ("LOGGING : OFF",1);
	}
  }
}

void main (void)
{
  unsigned char keys;
  bit disp_choice;
  power_heat1=0; power_heat2=0;		// No heatelement activated
  menu_item = 0;
  P1=0xff;							// Port 1 is input
  init_controller();
  init_rs232();
  printstr_rs232 ("Ovencontrol v1.0\n\r");
  init_keys();
  lcd_init();
  lcd_print_line ("Ovencontrol v1.0",0);
  lcd_print_line ("   Start up",1);
  wait_1ms();
  printstr_rs232 ("\n\r\n\rOvencontrol\n\r");
  if (is_24lc64_present()==0)
  {
    lcd_print_line ("   EEPROM BAD!",0);
	lcd_print_line (" replace 24lc64",1);
	while ((keys_get() & KEY_ENTER)==0) {}
  };
  LED1 = 1; LED2 = 1; LED3 = 1;
  tmp_log=0;										// Switch off the log-function to preserve EEPROM
  oven_overshoot = read_byte_24lc64 (0x00);			// Read the calibration-value
  if ((~oven_overshoot)!=read_byte_24lc64(0x01))	// Check if it is ok.
  {
    lcd_print_line ("Calibration",0);				// Else start the calibration-procedure
	lcd_print_line ("[ENT] start",1);

    while ((keys_get() & KEY_ENTER)==0) {}
	calibrate();
  }
  read_buffer_24lc64 (PROFILES_START_ADR,(unsigned char*) &profile,sizeof(profile));
  disp_choice=1;
  lcd_print_line ("ELEKTOR SMD-OVEN",0);
  menu_item=MENU_MIN;
  while (1)
  {
	keys = keys_get();						// get key-status
	if (keys & KEY_DOWN) ++menu_item;
	if (keys & KEY_UP) ++menu_item;
	if (menu_item>MENU_MAX) menu_item=MENU_MIN;
    if (menu_item<MENU_MIN) menu_item=MENU_MAX;    
    if (keys!=0) disp_choice=1;
   
    if (disp_choice!=0)
	switch (menu_item)
	{
	  case MENU_START : 
	    lcd_print_line ("->Start",1);
	  	break;
	  case MENU_EDIT  : 
	    lcd_print_line ("->Edit",1);
	  	break;
	  case MENU_LOG   :
	    lcd_print_line ("->Log",1);
		break;
	  case MENU_CAL   : 
	    lcd_print_line ("->Calibrate",1);
	  	break;
	  default : menu_item = MENU_MIN;
	}
	if (keys & KEY_ENTER)
	{
	  switch (menu_item)
	  {
	    case MENU_START:
		  control();
		  break;
	    case MENU_CAL :
	      calibrate();
		  break;
		case MENU_EDIT:
		  edit();
		  break;
		case MENU_LOG:
		  log();
		  break;
	    default:
	      break;
	  }
      lcd_print_line ("DONE",0);
      lcd_print_line ("press [ENT]",1);
      while ((keys_get() & KEY_ENTER)==0){};
      lcd_print_line ("ELEKTOR SMD-OVEN",0);
	  disp_choice=1;
	}
  }
}

