/* Name: main.c
 * Project: PowerSwitch based on AVR USB driver
 * Author: Christian Starkjohann
 * Creation Date: 2005-01-16
 * Tabsize: 4
 * Copyright: (c) 2005 by OBJECTIVE DEVELOPMENT Software GmbH
 * License: GNU GPL v2 (see License.txt) or proprietary (CommercialLicense.txt)
 * This Revision: $Id: main.c 523 2008-02-15 09:46:40Z cs $
 */

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>

#include "usbdrv/usbdrv.h"
#include "usbdrv/oddebug.h"
#include <util/delay.h>

#include "DSOneWire.h"
#include "LCD.h"



/*
This module implements an 8 bit parallel output controlled via USB. It is
intended to switch the power supply to computers and/or other electronic
devices.

Application examples:
- Rebooting computers located at the provider's site
- Remotely switch on/off of rarely used computers
- Rebooting other electronic equipment which is left unattended
- Control room heating from remote
*/

/*
USB pin
1 - vcc
2 - USB-
3 - USB+
4 - GND
5 - GND

USB D- INT1 PD3
USB D+ INT0 PD2
*/

/*
foglalt pinek
PB6 XTAL
PB7 XTAL

PC6 - RESET

PD2 USB+
PD3 USB-

PD4 -eletjel LED
PD5 -homerseklet lekerdezes LED

PB0-homerseklet lekerdezes

PC0 - USB CMD D0   LCD D0
PC1 - USB CMD D1   LCD D1
PC2 - USB CMD D2   LCD D2
PC3 - USB CMD D3   LCD D3
PC4 - USB CMD D4   LCD D4
PC5 - USB CMD D5   LCD D5

PB1 - USB CMD D6   LCD D6
PB2 - USB CMD D7   LCD D7
                   
PB3 - USB CMD D8   LCD E
PB4 - USB CMD D9   LCD RW
PB5 - USB CMD D10  LCD RS


*/

/*


PB2- LCD RS  1-DATA 0-Instruction code
PB1- LCD R/W 1-READ 0-WRITE
PC5- LCD E   1-Enab

PC0- LCD DB4
PC1- LCD DB5
PC2- LCD DB6
PC3- LCD DB7


*/

volatile uchar Temperature;//utoljara lekerdezett homerseklet
volatile uchar Status;     //utoljara lekerdezett DS1821 statusz
volatile unsigned int COUNT_REMAIN;//utoljara lekerdezett homerseklet 9 bit
volatile unsigned int COUNT_PER_C;//utoljara lekerdezett homerseklet 9 bit

volatile uchar UsbCmdSts;//usbFunctionSetup es main kozotti kommunikacio
#define USBCMD_START_QUERY_TEMP 0x01 //usb->main 0->1 homerseklet lekerdezes inditasa
                                    //main 1->0 homerseklet lekerdezve
#define USBCMD_LCD 0x02

#define USBCMD_LCD_LCDSet4bitMode		0
#define USBCMD_LCD_LCDClearDisplay		1
#define USBCMD_LCD_LCDReturnHome			2
#define USBCMD_LCD_LCDEntryModeSet		3
#define USBCMD_LCD_LCDDisplayControl		4
#define USBCMD_LCD_LCDCursorDisplayShift	5
#define USBCMD_LCD_LCDFunctionSet		6
#define USBCMD_LCD_LCDSetCGRAMAddress	7
#define USBCMD_LCD_LCDSetDDRAMAddress	8
#define USBCMD_LCD_LCDWriteDataToRAM		9
#define USBCMD_LCD_LCDReadBusyAndAddress	10
#define USBCMD_LCD_LCDReadDataFromRAM	11
#define USBCMD_LCD_LCDReadBusyAndAddress_ret	12
#define USBCMD_LCD_LCDReadDataFromRAM_ret	13

volatile unsigned char LCD_cmd;
volatile unsigned char LCD_val;

USB_PUBLIC uchar usbFunctionSetup(uchar data[8])
{
usbRequest_t    *rq = (void *)data;
static uchar    replyBuf[7];
usbMsgPtr = replyBuf;


	switch(rq->bRequest)
	{
		case 0://?elsz-e
			replyBuf[0]='H';replyBuf[1]='e';replyBuf[2]='l'
			;replyBuf[3]='l';replyBuf[4]='o'
			;replyBuf[5]=' ';replyBuf[6]='!';
			return 7;

		case 10://homerseklet lekerdezes inditasa
			UsbCmdSts|=USBCMD_START_QUERY_TEMP;
			return 0;
		case 11://homerseklet lekerdezes
			replyBuf[5]=UsbCmdSts&USBCMD_START_QUERY_TEMP;//lekerdezes allapota
			replyBuf[0]=Temperature;
			replyBuf[1]=COUNT_REMAIN>>8;
			replyBuf[2]=COUNT_REMAIN&0x00ff;
			replyBuf[3]=COUNT_PER_C>>8;
			replyBuf[4]=COUNT_PER_C&0x00ff;
			return 6;
		case 12://statusz lekerdezes
			replyBuf[0]=Status;
			return 1;

		case 20://LCD
			if(UsbCmdSts&USBCMD_LCD)
			{
				replyBuf[0]=0x01;//busy
				return 1;
			}
			switch(rq->wValue.bytes[0])
			{
				case USBCMD_LCD_LCDReadBusyAndAddress_ret:
					if(LCD_cmd==USBCMD_LCD_LCDReadBusyAndAddress)
					{
						replyBuf[0]=0;//no error
						replyBuf[1]=LCD_val;//ret val
						return 2;
					}
					else
					{
						replyBuf[0]=0x02;//not requsted
						return 1;
					}
				case USBCMD_LCD_LCDReadDataFromRAM_ret:
 					if(LCD_cmd==USBCMD_LCD_LCDReadDataFromRAM)
					{
						replyBuf[0]=0;//no error
						replyBuf[1]=LCD_val;//ret val
						return 2;
					}
					else
					{
						replyBuf[0]=0x02;//not requsted
						return 1;
					}

			}
			LCD_cmd=rq->wValue.bytes[0];
			LCD_val=rq->wValue.bytes[1];
			UsbCmdSts|=USBCMD_LCD;
			replyBuf[0]=0x00;//no error
			return 1;
        //PC0..5 DATA0..5
		//PB1..5 DATA6..10
		case 1://pin irany megadas I/O
			DDRC=(DDRC & 0xC0)|(rq->wValue.bytes[0] & 0x3F);	//DDRC0..5=DATA0..5
			DDRB=(DDRB & 0xC1) //11000001
				|((rq->wValue.bytes[0] & 0xC0)>>5) //DDRB1..2=DATA6..7	
				|((rq->wValue.bytes[1] & 0x07)<<3);//DDRB3..5=DATA8..10	
			return 0;
		case 2://pin irasa
			PORTC=(PORTC & 0xC0)|(rq->wValue.bytes[0] & 0x3F);	   //PORTC0..5=DATA0..5
			PORTB=(PORTB & 0xC1)
				|((rq->wValue.bytes[0] & 0xC0)>>5)//PORTB1..2=DATA6..7	
				|((rq->wValue.bytes[1] & 0x07)<<3);//DDRB3..5=DATA8..10	
			return 0;
		case 3://pin olvasasa
			replyBuf[0]=PINC & 0x3F;      //DATA0..5=PINC0..5
			replyBuf[0]|=(PINB & 0x06)<<5;//DATA6..7=PINB1..2  b110
			replyBuf[1]=(PINB & 0x38)>>3;//DATA8..10=PINB3..5  b111000
			return 2;
	}

    return 0;
}



/* allow some inter-device compatibility */
#if !defined TCCR0 && defined TCCR0B
#define TCCR0   TCCR0B
#endif
#if !defined TIFR && defined TIFR0
#define TIFR    TIFR0
#endif
void ShowTemp()
{
	char   tmpTemp;
	char t;
	char num;
	//Temperature-0.5+((COUNT_PER_C-COUNT_REMAIN)/COUNT_PER_C))

	LCDSetDDRAMAddress(0x40);//2. line

	tmpTemp=Temperature;
	//tizedes jegy
	t=((COUNT_PER_C-COUNT_REMAIN)*10)/COUNT_PER_C;

	if(t>=5){t-=5;}
		else {t+=5;tmpTemp--;}

	if(tmpTemp&0x80)
	{
		LCDWriteDataToRAM('-');
		tmpTemp*=-1;
	}
	else 
	{
		LCDWriteDataToRAM('+');
	}

	num=tmpTemp/100;
	if(num>0)LCDWriteDataToRAM('0'+num);

	num=(tmpTemp/10)%10;
	if(num>0)LCDWriteDataToRAM('0'+num);

	num=tmpTemp%10;
	LCDWriteDataToRAM('0'+num);

	LCDWriteDataToRAM('.');

	num=t%10;
	LCDWriteDataToRAM('0'+num);
}
int main(void)
{
uchar   i;
uchar c=0;
uchar doQueryT;
Temperature=0;
COUNT_REMAIN=0;
COUNT_PER_C=0;
UsbCmdSts=0;

SFIOR&=~PUD;//pull up off

	DDRB=0xff;//minden kimenet
	DDRC=0xff;//minden kimenet
	DDRD=0xff;//minden kimenet

    PORTC=0;
    PORTB=0;




	//eletjel LED villan
	PORTD = (1<<4);
	_delay_ms(100);
	PORTD &=~ (1<<4);

	//hoerzekelo init
	DSInitPin(0);//PB0

	PORTD = (1<<5);
	Status=DSQuerySts(0);
	if(!(Status&0x01))//1SHOT bit
	{
		DSStopQueryT(0);
		DSWriteSts(0,1);//1SHOT
		_delay_ms (200);


	}

	DSStartQueryT(0);
	
	i=0;
	do{
		_delay_ms (10);
		i++;
		Status=DSQuerySts(0);
	}while(!(Status&0x80)&&i<50);//DONE bit

	Temperature=DSQueryT(0);//PB0-s pin
	DSQueryTremain(0,&COUNT_REMAIN,&COUNT_PER_C);//PB0-s pin

	PORTD &=~ (1<<5);

	doQueryT=0;

//    wdt_enable(WDTO_1S);
    odDebugInit();

	LCDInit();
	LCDClearDisplay();
	LCDFunctionSet(LCD_CMD_N);
	LCDReturnHome();
	LCDDisplayControl(LCD_CMD_D|LCD_CMD_C|LCD_CMD_B);

	LCDWriteDataToRAM('H');
	LCDWriteDataToRAM('e');
	LCDWriteDataToRAM('l');
	LCDWriteDataToRAM('l');
	LCDWriteDataToRAM('o');

	ShowTemp();

/* no pullups on USB pins */
/* We fake an USB disconnect by pulling D+ and D- to 0 during reset. This is
 * necessary if we had a watchdog reset or brownout reset to notify the host
 * that it should re-enumerate the device. Otherwise the host's and device's
 * concept of the device-ID would be out of sync.
 */

//1- output 0-input
    DDRD = ~USBMASK;    /* set all pins as outputs except USB */
	PORTD = (1<<4);




//    computeOutputStatus();  /* set output status before we do the delay */
    usbDeviceDisconnect();  /* enforce re-enumeration, do this while interrupts are disabled! */
    i = 0;
    while(--i){         /* fake USB disconnect for > 500 ms */
        wdt_reset();
        _delay_ms(2);
    }
    usbDeviceConnect();
    TCCR0 = 5;          /*CS02..0=101 (5) set prescaler to 1/1024 */
    usbInit();
    sei();
    for(;;){    /* main event loop */
        wdt_reset();
        usbPoll();
		if(UsbCmdSts&USBCMD_LCD)
		{
			switch(LCD_cmd)
			{
				case USBCMD_LCD_LCDSet4bitMode:LCDSet4bitMode();break;
				case USBCMD_LCD_LCDClearDisplay:LCDClearDisplay();break;
				case USBCMD_LCD_LCDReturnHome:LCDReturnHome();break;
				case USBCMD_LCD_LCDEntryModeSet:LCDEntryModeSet(LCD_val);break;
				case USBCMD_LCD_LCDDisplayControl:LCDDisplayControl(LCD_val);break;
				case USBCMD_LCD_LCDCursorDisplayShift:LCDCursorDisplayShift(LCD_val);break;
				case USBCMD_LCD_LCDFunctionSet:LCDFunctionSet(LCD_val);break;
				case USBCMD_LCD_LCDSetCGRAMAddress:LCDSetCGRAMAddress(LCD_val);break;
				case USBCMD_LCD_LCDSetDDRAMAddress:LCDSetDDRAMAddress(LCD_val);break;
				case USBCMD_LCD_LCDWriteDataToRAM:LCDWriteDataToRAM(LCD_val);break;
				case USBCMD_LCD_LCDReadBusyAndAddress:LCD_val=LCDReadBusyAndAddress();break;
				case USBCMD_LCD_LCDReadDataFromRAM:LCD_val=LCDReadDataFromRAM();break;

			}
			UsbCmdSts&=~USBCMD_LCD;
		}

        if(TIFR & (1 << TOV0)){
            TIFR |= 1 << TOV0;  /* clear pending flag */


				if(doQueryT==1)
				{
					cli();
					Status=DSQuerySts(0);
					sei();
					if(Status&0x80)//DONE bit
					{
						cli();
						Temperature=DSQueryT(0);//PB0-s pin

						DSQueryTremain(0,&COUNT_REMAIN,&COUNT_PER_C);//PB0-s pin
						
						UsbCmdSts&=~USBCMD_START_QUERY_TEMP;//lekerdezes kessz
						sei();

						doQueryT=0;
						PORTD&=~(1<<5);//1 s villogas

						//ertek kiirasa

						ShowTemp();

					}

				}
				if(doQueryT==0&&(UsbCmdSts&USBCMD_START_QUERY_TEMP))
				{
					cli();
					DSStartQueryT(0);
					sei();
					doQueryT=1;
					PORTD|=(1<<5);//1 s villogas

				}


			if(++c==30)
			{
				c=0;
				PORTD^=(1<<4);//1 s villogas


			}
//            timerInterrupt();
        }
    }
    return 0;
}




