;***************************************************************************
;* U S B   S T A C K   F O R   T H E   A V R   F A M I L Y
;* 
;* File Name            :"MJoy.asm"
;* Title                :MJoy  - USB Joystick based on USB stack
;* Date                 :1.17.2004
;* Version              :1.2
;* Target MCU           :ATmega8
;* Assembler			:Atmel AVR Studio
;* AUTHOR		:	Mindaugas Milasauskas	
;*					Lithuania
;*					mindaug@mindaugas.com
;*					http://www.mindaugas.com
;* Based on code of: 
;*					Ing. Igor Cesko
;* 			 		Slovakia
;* 				 	cesko@internet.sk
;* 			 		http://www.cesko.host.sk
;*
;* DESCRIPTION:
;*  USB protocol implementation into MCU with noUSB interface:
;*  Device:
;*  MJoy USB Joystick
;*
;* The timing is adapted for 12 MHz crystal
;*
;*
;*	Release Notes:
;*
;*	v1.2:	Second public release
;*			6 axis (4 x 10 bit + 2 x 8 bit), 24 buttons, 1 hat (uses 4 buttons)
;*			Added autocalibration
;*			Changed layout of HID descriptor
;*			Changed USB Vendor ID to make MJoy appear on top of the game devices list
;*			Optimized HatSwitch Code
;*			Rudder and Throttle axis ar now 10bit
;*			Changed axis Dial, Slider to Rx, Z
;*			Optimized partially non-blocking ADC reading
;*
;*
;*	v1.1:	First public release
;*			6 axis - 2x10bit axis + 4x8bit axis
;*			24 buttons
;*			1 hat (uses 4 buttons)
;*			
;*	v1.0:	Very first lab release with 2x10bit axis and 4 buttons
;*
;*
;***************************************************************************
.include "m8def.inc"
.equ	UCR			=UCSRB
.equ	UBRR			=UBRRL
.equ	EEAR			=EEARL
.equ	USR			=UCSRA
.equ	E2END			=127
.equ	RAMSIZE			=159
.equ	RAMEND_MJ			=255

.equ	inputport		=PINB
.equ	outputport		=PORTB
.equ	USBdirection		=DDRB
.equ	DATAplus		=1		;signal D+ on PB1
.equ	DATAminus		=0		;signal D- on PB0 - give on this pin pull-up 1.5kOhm
.equ	USBpinmask		=0b11111100	;mask low 2 bit (D+,D-) on PB
.equ	USBpinmaskDplus		=~(1<<DATAplus)	;mask D+ bit on PB
.equ	USBpinmaskDminus	=~(1<<DATAminus);mask D- bit on PB


.equ	SOPbyte			=0b10000000	;Start of Packet byte
.equ	DATA0PID		=0b11000011	;PID for DATA0 field
.equ	DATA1PID		=0b01001011	;PID for DATA1 field
.equ	OUTPID			=0b11100001	;PID for OUT field
.equ	INPID			=0b01101001	;PID for IN field
.equ	SOFPID			=0b10100101	;PID for SOF field
.equ	SETUPPID		=0b00101101	;PID for SETUP field
.equ	ACKPID			=0b11010010	;PID for ACK field
.equ	NAKPID			=0b01011010	;PID for NAK field
.equ	STALLPID		=0b00011110	;PID for STALL field
.equ	PREPID			=0b00111100	;PID for FOR field

.equ	nSOPbyte		=0b00000001	;Start of Packet byte - reverse order
.equ	nDATA0PID		=0b11000011	;PID for DATA0 field - reverse order
.equ	nDATA1PID		=0b11010010	;PID for DATA1 field - reverse order
.equ	nOUTPID			=0b10000111	;PID for OUT field - reverse order
.equ	nINPID			=0b10010110	;PID for IN field - reverse order
.equ	nSOFPID			=0b10100101	;PID for SOF field - reverse order
.equ	nSETUPPID		=0b10110100	;PID for SETUP field - reverse order
.equ	nACKPID			=0b01001011	;PID for ACK field - reverse order
.equ	nNAKPID			=0b01011010	;PID for NAK field - reverse order
.equ	nSTALLPID		=0b01111000	;PID for STALL field - reverse order
.equ	nPREPID			=0b00111100	;PID for FOR field - reverse order

.equ	nNRZITokenPID		=~0b10000000	;PID mask for Token packet (IN,OUT,SOF,SETUP) - reverse order NRZI
.equ	nNRZISOPbyte		=~0b10101011	;Start of Packet byte - reverse order NRZI
.equ	nNRZIDATA0PID		=~0b11010111	;PID for DATA0 field - reverse order NRZI
.equ	nNRZIDATA1PID		=~0b11001001	;PID for DATA1 field - reverse order NRZI
.equ	nNRZIOUTPID		=~0b10101111	;PID for OUT field - reverse order NRZI
.equ	nNRZIINPID		=~0b10110001	;PID for IN field - reverse order NRZI
.equ	nNRZISOFPID		=~0b10010011	;PID for SOF field - reverse order NRZI
.equ	nNRZISETUPPID		=~0b10001101	;PID for SETUP field - reverse order NRZI
.equ	nNRZIACKPID		=~0b00100111	;PID for ACK field - reverse order NRZI
.equ	nNRZINAKPID		=~0b00111001	;PID for NAK field - reverse order NRZI
.equ	nNRZISTALLPID		=~0b00000111	;PID for STALL field - reverse order NRZI
.equ	nNRZIPREPID		=~0b01111101	;PID for FOR field - reverse order NRZI
.equ	nNRZIADDR0		=~0b01010101	;Address = 0 - reverse order NRZI

						;status bytes - State
.equ	BaseState		=0		;
.equ	SetupState		=1		;
.equ	InState			=2		;
.equ	OutState		=3		;
.equ	SOFState		=4		;
.equ	DataState		=5		;

						;Flags of action
.equ	DoNone					=0
.equ	DoReceiveOutData			=1
.equ	DoReceiveSetupData			=2
.equ	DoPrepareOutContinuousBuffer		=3
.equ	DoReadySendAnswer			=4
.equ	DoPrepareJoystickAnswer			=5
.equ	DoReadySendJoystickAnswer			=6

			; Joystick flags
.equ	JoystickDataRequest		=1
.equ	JoystickDataRequestBit		=0
.equ	JoystickDataReady		=2
.equ	JoystickDataReadyBit		=1
.equ	JoystickDataProcessing	=4
.equ	JoystickDataProcessingBit	=2
.equ	JoystickLastDataPID		=8
.equ	JoystickLastDataPIDBit		=3
.equ	JoystickReportID		=0b00010000
.equ	JoystickReportIDBit			=4

.equ	CRC16poly		=0b1000000000000101	;CRC16 polynomial

.equ	Sym10StructSize	=12
.equ	Sym10EEStructSize	=4
.equ	Asym8StructSize =8
.equ	Asym8EEStructSize =4

.equ	XAxisChannel	= 0
.equ	YAxisChannel	= 1
.equ	RudderAxisChannel	= 2
.equ	ThrottleAxisChannel	= 3
.equ	ZAxisChannel		= 4
.equ	RxAxisChannel	= 5

.equ	MAXUSBBYTES		=14			;maximum bytes in USB input message
.equ	NumberOfFirstBits	=10			;how many first bits allowed be longer
.equ	NoFirstBitsTimerOffset	=256-12800*12/1024	;Timeout 12.8ms (12800us) to terminate after firsts bits

.equ	InputBufferBegin	=RAMEND_MJ-RAMSIZE				;compare of receiving shift buffer
.equ	InputShiftBufferBegin	=InputBufferBegin+MAXUSBBYTES		;compare of receiving buffera

.equ	OutputBufferBegin	=RAMEND_MJ-MAXUSBBYTES-2	;compare of transmitting buffer
.equ	AckBufferBegin		=OutputBufferBegin-3	;compare of transmitting buffer Ack
.equ	NakBufferBegin		=AckBufferBegin-3	;compare of transmitting buffer Nak
.equ	ConfigByte		=NakBufferBegin-1	;0=unconfigured state
.equ	AnswerArray		=ConfigByte-8		;8 byte answer array
.equ	JoystickBufferBegin	= AnswerArray - MAXUSBBYTES
.equ	JoyOutputBufferLength = JoystickBufferBegin - 1
.equ	JoyOutBitStuffNumber = JoyOutputBufferLength - 1
.equ	BkpOutputBufferLength = JoyOutBitStuffNumber - 1
.equ	BkpOutBitStuffNumber = BkpOutputBufferLength - 1
.equ	JoyVal = BkpOutBitStuffNumber - 1
.equ	ADCChanStarted		=JoyVal - 1
.equ	ADCValueH		=ADCChanStarted - 1
.equ	ADCValueL		=ADCValueH -1

.equ	RegistersBackup = ADCValueL-32

.equ	EEPROMFlags = RegistersBackup-1

.equ	StackBegin		=EEPROMFlags-1		;low reservoir (stack is big cca 54 byte)


.equ	XAxisRAM 			=RAMEND - Sym10StructSize
.equ	YAxisRAM 			=XAxisRAM - Sym10StructSize
.equ	RudderAxisRAM 		=YAxisRAM - Sym10StructSize
.equ	ThrottleAxisRAM 	=RudderAxisRAM - Sym10StructSize
.equ	ZAxisRAM 			=ThrottleAxisRAM - Asym8StructSize
.equ	RxAxisRAM 			=ZAxisRAM - Asym8StructSize



.def	JoystickFlags		=R1		; Endpoint 1 interrupt status flags for joystick reports
.def	backupbitcount		=R2		;backup bitcount register in INT0 disconnected
.def	RAMread			=R3		;if reading from SRAM
.def	backupSREGTimer		=R4		;backup Flag register in Timer interrupt
.def	backupSREG		=R5		;backup Flag register in INT0 interrupt
.def	ACC			=R6		;accumulator
.def	lastBitstufNumber	=R7		;position in bitstuffing
.def	OutBitStuffNumber	=R8		;how many bits to send last byte - bitstuffing
.def	BitStuffInOut		=R9		;if insertion or deleting of bitstuffing
.def	TotalBytesToSend	=R10		;how many bytes to send
.def	TransmitPart		=R11		;order number of transmitting part
.def	InputBufferLength	=R12		;length prepared in input USB buffer
.def	OutputBufferLength	=R13		;length answers prepared in USB buffer
.def	MyUpdatedAddress	=R14		;my USB address for update
.def	MyAddress		=R15		;my USB address


.def	ActionFlag		=R16		;what to do in main program loop
.def	temp3			=R17		;temporary register
.def	temp2			=R18		;temporary register
.def	temp1			=R19		;temporary register
.def	temp0			=R20		;temporary register
.def	bitcount		=R21		;counter of bits in byte
.def	ByteCount		=R22		;counter of maximum number of received bytes
.def	inputbuf		=R23		;receiver register 
.def	shiftbuf		=R24		;shift receiving register 
.def	State			=R25		;state byte of status of state machine
;.def	XL				=R26		;XL register - pointer to buffer of received IR codes
;.def	XH				=R27
.def	USBBufptrY		=R28		;YL register - pointer to USB buffer input/output
.def	ROMBufptrZ		=R30		;ZL register - pointer to buffer of ROM data


;requirements on descriptors
.equ	GET_STATUS		=0
.equ	CLEAR_FEATURE		=1
.equ	SET_FEATURE		=3
.equ	SET_ADDRESS		=5
.equ	GET_DESCRIPTOR		=6
.equ	SET_DESCRIPTOR		=7
.equ	GET_CONFIGURATION	=8
.equ	SET_CONFIGURATION	=9
.equ	GET_INTERFACE		=10
.equ	SET_INTERFACE		=11
.equ	SYNCH_FRAME		=12

; Class requests
.equ 	GET_REPORT		=1
.equ 	GET_IDLE		=2
.equ 	GET_PROTOCOL	=3
.equ 	SET_REPORT		=9
.equ 	SET_IDLE		=10
.equ 	SET_PROTOCOL	=11

;Standard descriptor types
.equ	DEVICE			=1
.equ	CONFIGURATION		=2
.equ	STRING			=3
.equ	INTERFACE		=4
.equ	ENDPOINT		=5

; Class Descriptor Types
.equ	CLASS_HID		=0x21
.equ	CLASS_Report	=0x22
.equ	CLASS_Physical	=0x23

;databits
.equ	DataBits5		=0
.equ	DataBits6		=1
.equ	DataBits7		=2
.equ	DataBits8		=3

;parity
.equ	ParityNone		=0
.equ	ParityOdd		=1
.equ	ParityEven		=2
.equ	ParityMark		=3
.equ	ParitySpace		=4

;stopbits
.equ	StopBit1		=0
.equ	StopBit2		=1

;user function start number
.equ	USER_FNC_NUMBER		=100


.equ	UpdEEThd	= 1500	; 15 seconds to wait before EEPROM write	
							; this is the threshold determining how many cycles to pass before
							; writing axis data to EEPROM. It determines the delay before 
							; writing which is Delay = CyclePeriod * UpdEEThd

; The structure of Joystick Axis data; All values are offsets
.equ	s_UpdateCntH	=0		; Update counter to avoid too many writes to EEPROM
.equ	s_UpdateCntL	=1

.equ	s_EEPROMStructStart		=2		; 16 bit unsigned minimum registered ADC input value
.equ	s_MinH 			=2		; 16 bit unsigned minimum registered ADC input value
.equ	s_MinL 			=3		; 
.equ	s_MaxH 			=4		; 16 bit unsigned maximum registered ADC input value
.equ	s_MaxL 			=5		; 
.equ	s_CoefH 		=6		; 16 bit unsigned asymmetric axis range coeficient
.equ	s_CoefL 		=7

.equ	s_NegCoefH 		=6		; 16 bit unsigned axis negative range coeficient
.equ	s_NegCoefL 		=7
.equ	s_CenterH 		=8		; 16 bit unsigned axis center / offset value
.equ	s_CenterL 		=9		;		 ( not used for asymmetric axes )
.equ	s_PosCoefH 		=10		; 16 bit unsigned axis negative range coeficient
.equ	s_PosCoefL 		=11		;		 ( not used for asymmetric axes )





;------------------------------------------------------------------------------------------
;********************************************************************
;* ;* Interrupt table
;********************************************************************
.cseg
;------------------------------------------------------------------------------------------
.org 0						;after reset
		rjmp	reset
;------------------------------------------------------------------------------------------
.org INT0addr					;external interrupt INT0
		rjmp	INT0handler
;------------------------------------------------------------------------------------------

.org ERDYaddr
		reti
;------------------------------------------------------------------------------------------
;********************************************************************
;*  Init program
;********************************************************************
;------------------------------------------------------------------------------------------
reset:			;initialization of processor and variables to right values
		ldi	temp0,StackBegin	;initialization of stack
		out	SPL,temp0

		clr	XH				;RS232 pointer
		clr	YH				;USB pointer
		clr	ZH				;ROM pointer
		clr	JoystickFlags
		ldi		temp0,JoystickDataReady
		or		JoystickFlags,temp0 


		clr	MyUpdatedAddress	;new address USB -  non-decoded
		rcall	InitACKBufffer		;initialization of ACK buffer
		rcall	InitNAKBufffer		;initialization of NAK buffer
		rcall	InitJoystickBufffer		;initialization of Joystick buffer

		rcall	USBReset		;initialization of USB addresses
		
		ldi	temp0,0b00000100	;set pull-up on PORTB
		out	PORTB,temp0
		ldi	temp0,0b11000000	;set pull-up on PORTC
		out	PORTC,temp0
		ldi	temp0,0b01111011	;set pull-up on PORTD
		out	PORTD,temp0

		ldi	temp0,0b00000000	;set input on PORTD
		out	DDRD,temp0

		clr	temp0			;
		out	EEARH,temp0		;zero EEPROM index

	; prepare ADC
		ldi	temp0, 0b10000110	;ADC enable
								; no start yet
								; Single Conversion mode
								; no interrupts
								;set ADC prescaler to 12MHz / 64 = 187.5kHz
		out ADCSRA,temp0	

		ldi	temp0, 0b01000000	;AVCC refernce, PortC 0 - X
								; clear ADLAR for 10 bit conversion
		out ADMUX, temp0

;-- initialize all joystick axes
		;check if recalibration is needed

		ldi		temp3, 0
	; Read row 1
		sbi		DDRB,3
		cbi		PORTB,3
		rcall	ReadButtonsRow
		sbi		PORTB,3
		cbi		DDRB,3			;return to tristate
		add		temp3,temp0

	; Read row 2
		sbi		DDRB,4
		cbi		PORTB,4
		rcall	ReadButtonsRow
		sbi		PORTB,4
		cbi		DDRB,4			;return to tristate
		add		temp3,temp0

	;Read row 3
		sbi		DDRB,5
		cbi		PORTB,5
		rcall	ReadButtonsRow
		sbi		PORTB,5
		cbi		DDRB,5			;return to tristate
		add		temp3,temp0


	;Read row 4
		sbi		DDRD,7
		cbi		PORTD,7
		rcall	ReadButtonsRow
		sbi		PORTD,7
		cbi		DDRC,7			;return to tristate
		add		temp3,temp0

		tst		temp3
		brne	RecalibNeeded
		ldi		temp0, 0
		rjmp	Cont2
RecalibNeeded:
		ldi		temp0, 1
Cont2:
		sts		EEPROMFlags, temp0
		
		cbi EECR,EERIE
		rcall	InitJoystickAxis

		
		ldi	temp0,0x0F		;INT0 - respond to leading edge
		out	MCUCR,temp0		;
		ldi	temp0,1<<INT0		;enable external interrupt INT0
		out	GIMSK,temp0
;------------------------------------------------------------------------------------------
;********************************************************************
;*  Main program
;********************************************************************
		sei					;enable interrupts globally
Main:
		sbis	inputport,DATAminus	;waiting till change D- to 0
		rjmp	CheckUSBReset		;and check, if isn't USB reset

		cpi	ActionFlag,DoReceiveSetupData
		breq	ProcReceiveSetupData
		cpi	ActionFlag,DoPrepareOutContinuousBuffer
		breq	ProcPrepareOutContinuousBuffer
		sbrc	JoystickFlags,JoystickDataRequestBit
		rcall	ProcJoystickRequest
		
		rjmp	Main

CheckUSBReset:
		ldi	temp0,255		;counter duration of reset (according to specification is that cca 10ms - here is cca 100us)
WaitForUSBReset:
		sbic	inputport,DATAminus	;waiting till change D+ to 0
		rjmp	Main
		dec	temp0
		brne	WaitForUSBReset
		rcall	USBReset
		rjmp	Main

ProcPrepareOutContinuousBuffer:
		rcall	PrepareOutContinuousBuffer	;prepare next sequence of answer to buffer
		ldi	ActionFlag,DoReadySendAnswer
		rjmp	Main
ProcReceiveSetupData:
		ldi	USBBufptrY,InputBufferBegin	;pointer to begin of receiving buffer
		mov	ByteCount,InputBufferLength	;length of input buffer
		rcall	DecodeNRZI		;transfer NRZI coding to bits
		rcall	MirrorInBufferBytes	;invert bits order in bytes
		rcall	BitStuff		;removal of bitstuffing
		;rcall	CheckCRCIn		;rcall	CheckCRCIn		;check CRC
		rcall	PrepareUSBOutAnswer	;prepare answers to transmitting buffer
		ldi	ActionFlag,DoReadySendAnswer
		rjmp	Main


;********************************************************************
;*  Joystick Initialization Routine
;********************************************************************

InitJoystickAxis:

		call	BackupUpperRegisters

		ldi r16, 0xFF
		sts	ADCChanStarted,r16	; reset last ADC  channel number

		ldi r16, XAxisChannel	;
		call ReadADC10
		ldi	ZH, high(XAxisRAM)	
		ldi	ZL,  low(XAxisRAM)
		ldi	YH, high(XAxisEE)	
		ldi	YL,  low(XAxisEE)
		call Init10SymAxis

		ldi r16, YAxisChannel	;
		call ReadADC10
		ldi	ZH, high(YAxisRAM)	
		ldi	ZL,  low(YAxisRAM)
		ldi	YH, high(YAxisEE)	
		ldi	YL,  low(YAxisEE)
		call Init10SymAxis

		ldi r16, RudderAxisChannel	;
		call ReadADC10
		ldi	ZH, high(RudderAxisRAM)	
		ldi	ZL,  low(RudderAxisRAM)
		ldi	YH, high(RudderAxisEE)	
		ldi	YL,  low(RudderAxisEE)
		call Init10SymAxis

		ldi	ZH, high(ThrottleAxisRAM)	
		ldi	ZL,  low(ThrottleAxisRAM)
		ldi	YH, high(ThrottleAxisEE)	
		ldi	YL,  low(ThrottleAxisEE)
		call Init8AsymAxis

		ldi	ZH, high(ZAxisRAM)	
		ldi	ZL,  low(ZAxisRAM)
		ldi	YH, high(ZAxisEE)	
		ldi	YL,  low(ZAxisEE)
		call Init8AsymAxis

		ldi	ZH, high(RxAxisRAM)	
		ldi	ZL,  low(RxAxisRAM)
		ldi	YH, high(RxAxisEE)	
		ldi	YL,  low(RxAxisEE)
		call Init8AsymAxis


		call RestoreUpperRegisters

		ret

;********************************************************************
;*  Joystick Calibration Ranges Update Routine
;********************************************************************
CheckAxisRanges:
		call	BackupUpperRegisters
		
; ----- check if some axis range was changed and is needed to be written to EEPROM
		ldi	ZH, high(XAxisRAM)	
		ldi	ZL,  low(XAxisRAM)
		ldi	YH, high(XAxisEE)	
		ldi	YL,  low(XAxisEE)
		call CheckAxisRangeUpdate
		breq CheckRangeEnd			; skip everything else if we have written to eeprom


		ldi	ZH, high(YAxisRAM)	
		ldi	ZL,  low(YAxisRAM)
		ldi	YH, high(YAxisEE)	
		ldi	YL,  low(YAxisEE)
		call CheckAxisRangeUpdate
		breq CheckRangeEnd


		ldi	ZH, high(RudderAxisRAM)	
		ldi	ZL,  low(RudderAxisRAM)
		ldi	YH, high(RudderAxisEE)	
		ldi	YL,  low(RudderAxisEE)
		call CheckAxisRangeUpdate
		breq CheckRangeEnd

		ldi	ZH, high(ThrottleAxisRAM)	
		ldi	ZL,  low(ThrottleAxisRAM)
		ldi	YH, high(ThrottleAxisEE)	
		ldi	YL,  low(ThrottleAxisEE)
		call CheckAxisRangeUpdate
		breq CheckRangeEnd

		ldi	ZH, high(ZAxisRAM)	
		ldi	ZL,  low(ZAxisRAM)
		ldi	YH, high(ZAxisEE)	
		ldi	YL,  low(ZAxisEE)
		call CheckAxisRangeUpdate
		breq CheckRangeEnd

		ldi	ZH, high(RxAxisRAM)	
		ldi	ZL,  low(RxAxisRAM)
		ldi	YH, high(RxAxisEE)	
		ldi	YL,  low(RxAxisEE)
		call CheckAxisRangeUpdate
		breq CheckRangeEnd

rjmp	CheckRangeEnd


CheckRangeEnd:
		call 	RestoreUpperRegisters
		ret

;*********************************************************************		
;* Check if axis range was updated and write updated values to EEPROM if needed
;*
;* Input: 	Z 		- Axis structure RAM address
;* 			Y		- Axis structure EEPROM address
;*
;* Registers used:
;*			r16, r17, Y, Z
;*********************************************************************

CheckAxisRangeUpdate:
		ldd r25, Z + s_UpdateCntH
		ldd r24, Z + s_UpdateCntL
		adiw	r25:r24,0	; just to set the flags
		breq	Ret10SARU	; if no update then return
		adiw	r25:r24,1	; increment update delay counter
		std Z + s_UpdateCntH, r25
		std Z + s_UpdateCntL, r24

		ldi	r19, high(UpdEEThd)
		ldi r18, low(UpdEEThd)
		cp	r24, r18			; check if value exceeds  MINIMUM range
		cpc	r25, r19
		brlo	Ret10SARU		; return if threshold not reached
		; reset the update counter to 0
		ldi	r16, 0
		std Z + s_UpdateCntL, r16	; store the 0
		std Z + s_UpdateCntH, r16

		cli		; disable interrupts
		; write axis structure to EEPROM
		movw	r19:r18, ZH:ZL 	; backup structure pointer
		adiw	ZH:ZL, s_EEPROMStructStart	; point to start of EEPROM part
		ldi	r17, Sym10EEStructSize
		call SaveStructToEEPROM
		sez		; set to mark that we have written to eeprom
		sei		;enable interrupts
		ret
Ret10SARU:
		clz		
		sei		;enable interrupts
		ret



;*********************************************************************		
;* Load structure from EEPROM to Data memory
;*
;* Input: 	Y 		- EEPROM address (source)
;*			Z 		- Data address (destination)
;*			r17		- Number of bytes to load
;* 
;*
;* Registers used:
;*			r16, r17, Y, Z
;*********************************************************************
LoadStructFromEEPROM:
			
		sbic EECR,EEWE	; Wait for completion of previous write ( if there was one)
		rjmp LoadStructFromEEPROM
		out	EEARL,YL		;set the address EEPROM Lo
		out	EEARH,YH		;set the address EEPROM Hi
		sbi	EECR,EERE		;read EEPROM to register EEDR
		in	R16,EEDR			;load from EEDR to R0
		st	Z+,R16			;R0 save to buffer and increment buffer
		adiw	YH:YL,1			;increment index to EEPROM
		dec	R17		;till are not all bytes
		brne	LoadStructFromEEPROM;then load next
		ret

;*******************************************************************		
; Save structure from Data memory to EEPROM
;
; Input: 	Y 		- EEPROM address (destination)
;			Z 		- Data address (source)
;			r17		- Number of bytes to save
; 
;
; Registers used:
;			r16, r17, Y, Z
;*********************************************************************
SaveStructToEEPROM:
		sbic EECR,EEWE		; Wait for completion of previous write
		rjmp SaveStructToEEPROM
		out	EEARL,YL		;set the address EEPROM Lo
		out	EEARH,YH		;set the address EEPROM Hi
		ld	R16, Z+			;load value from RAM to R16
		out EEDR,r16		; Write data (r16) to data register
		sbi EECR,EEMWE		; Write logical one to EEMWE
		sbi EECR,EEWE		; Start eeprom write by setting EEWE
		adiw	YH:YL,1			;increment index to EEPROM
		dec	R17				;till are not all bytes
		brne	SaveStructToEEPROM	;then load next
		ret




;********************************************************************
;*  Joystick Requests Processing Routine
;********************************************************************

ProcJoystickRequest:

		call BackupUpperRegisters
	; clear ready flag to avoid data conflict
		mov		temp0, JoystickFlags
		ldi		temp0,0xFF
		andi		temp0,~JoystickDataReady
		andi	temp0,~JoystickDataRequest ; clear request flag to avoid call on next cycle
		and		JoystickFlags,temp0 

		sbrc	JoystickFlags,JoystickReportIDBit
		rjmp	ProcessJoystickReport_2

ProcessJoystickReport_1:
		;save report ID
		ldi		temp0, 1
		sts		JoystickBufferBegin+2,temp0 

;;;;;;;;;;;;;;;;;;;;;;;;
; Read value from ADC X channel

		ldi		r18, XAxisChannel	; initiate ADC conversion for next axis
		rcall	StartADC10
		ldi		r16, XAxisChannel
		rcall	ReadADC10
		ldi		r18, YAxisChannel	; initiate ADC conversion for next axis
		rcall	StartADC10
		ldi	ZH, high(XAxisRAM)	
		ldi	ZL,  low(XAxisRAM)
		call Process10SymAxis

	; Store X value
		sts		JoystickBufferBegin+3,r16
		sts		JoystickBufferBegin+4,r17


;;;;;;;;;;;;;;;;;
; Read value from ADC Y channel
		ldi		r16, YAxisChannel
		rcall	ReadADC10
		ldi		r18, RudderAxisChannel	; initiate ADC conversion for next axis
		rcall	StartADC10
		ldi	ZH, high(YAxisRAM)	
		ldi	ZL,  low(YAxisRAM)
		call Process10SymAxis

	; shift for Y value
		lsl		r16
		rol		r17
		lsl		r16
		rol		r17
		lds		r18, JoystickBufferBegin+4 ; to combine with X
		or		r18, r16
		sts		JoystickBufferBegin+4,r18
		sts		JoystickBufferBegin+5,r17

;;;;;;;;;;;;;;;;;
; Read value from ADC Rudder channel
		ldi		r16, RudderAxisChannel
		rcall	ReadADC10
		ldi		r18, ThrottleAxisChannel	; initiate ADC conversion for next axis
		rcall	StartADC10
		ldi	ZH, high(RudderAxisRAM)	
		ldi	ZL,  low(RudderAxisRAM)
		call Process10SymAxis
		mov		temp1, r17
		mov		temp0, r16

	; shift for Rudder value
		
		lsl		r16
		rol		r17
		lsl		r16
		rol		r17
		lsl		r16
		rol		r17
		lsl		r16
		rol		r17
		lds		r18, JoystickBufferBegin+5 ; to combine with Y
		or		r18, r16
		sts		JoystickBufferBegin+5,r18
		sts		JoystickBufferBegin+6,r17
	

;;;;;;;;;;;;;;;;;
; Read value from ADC Throttle channel
		ldi		r16, ThrottleAxisChannel
		rcall	ReadADC10
		ldi		r18, ZAxisChannel	; initiate ADC conversion for next axis
		rcall	StartADC10
		ldi	ZH, high(ThrottleAxisRAM)	
		ldi	ZL,  low(ThrottleAxisRAM)
		call Process10AsymAxis

		mov		temp1, r17
		mov		temp0, r16

		lsl		temp0
		rol		temp1
		lsl		temp0
		rol		temp1
		lsl		temp0
		rol		temp1
		lsl		temp0
		rol		temp1
		lsl		temp0
		rol		temp1
		lsl		temp0
		rol		temp1

		lds		temp3, JoystickBufferBegin+6 ; to combine with Rudder
		or		temp3, temp0
		sts		JoystickBufferBegin+6,temp3
		sts		JoystickBufferBegin+7,temp1

;;;;;;;;;;;;;;;;;
; Read value from Z ADC channel
		ldi		r16, ZAxisChannel
		rcall	ReadADC10
		ldi		r18, RxAxisChannel	; initiate ADC conversion for next axis
		rcall	StartADC10
		ldi	ZH, high(ZAxisRAM)	
		ldi	ZL,  low(ZAxisRAM)
		call Process8AsymAxis
		sts		JoystickBufferBegin+8,r16 


;;;;;;;;;;;;;;;;;
; Read value from Rx ADC channel
		ldi		r16, RxAxisChannel
		rcall	ReadADC10
		ldi	ZH, high(RxAxisRAM)	
		ldi	ZL,  low(RxAxisRAM)
		call Process8AsymAxis
		sts		JoystickBufferBegin+9,r16 
		call 	RestoreUpperRegisters
		ldi		temp3,JoystickReport1Size		;Joystick report size
		rjmp	SendJoystickReport

ProcessJoystickReport_2:
		;save report ID
		ldi		temp0, 2
		sts		JoystickBufferBegin+2,temp0 	; write Report ID

	; Read row 1
		sbi		DDRB,3
		cbi		PORTB,3
		rcall	ReadButtonsRow
		sbi		PORTB,3
		cbi		DDRB,3			;return to tristate
		mov		temp3,temp0

	; Read row 2
		sbi		DDRB,4
		cbi		PORTB,4
		rcall	ReadButtonsRow
		sbi		PORTB,4
		cbi		DDRB,4			;return to tristate
		ror		temp0			; shift down
		ror		temp1			; get lowest bit to top
		andi	temp1,0b10000000
		or		temp3, temp1
		sts		JoystickBufferBegin+3,temp3 
		andi	temp0,0b00111111
		mov		temp3,temp0

	;Read row 3
		sbi		DDRB,5
		cbi		PORTB,5
		rcall	ReadButtonsRow
		sbi		PORTB,5
		cbi		DDRB,5			;return to tristate
		ror		temp0			; shift down
		ror		temp1			; get lowest bit to top
		ror		temp0			; shift down
		ror		temp1			; get lowest bit to top
		andi	temp1,0b11000000
		or		temp3, temp1
		sts		JoystickBufferBegin+4,temp3 
		andi	temp0,0b00011111
		mov		temp3,temp0

	;Read row 4
		sbi		DDRD,7
		cbi		PORTD,7
		rcall	ReadButtonsRow
		sbi		PORTD,7
		cbi		DDRC,7			;return to tristate
		mov		temp2, temp0
		ror		temp0			; shift down
		ror		temp1			; get lowest bit to top
		ror		temp0			; shift down
		ror		temp1			; get lowest bit to top
		ror		temp0			; shift down
		ror		temp1			; get lowest bit to top
		andi	temp1,0b11100000
		or		temp3, temp1
		sts		JoystickBufferBegin+5,temp3 

;;;;;;;;;;;;;;;;;;;;;;;;;

; Process Hat switch
		mov		temp0, temp2
		lsl		temp0
		swap	temp0
		andi	temp0, 0b00001111
		ldi		ZH, high(HatValues << 1)
		ldi		ZL, low(HatValues << 1)
		ldi		temp1, 0
		add		ZL, temp0
		adc		ZH, temp1
		lpm		temp0, Z		; read value from memory mapping
		sts		JoystickBufferBegin+6,temp0 
		call 	RestoreUpperRegisters
		ldi		temp3,JoystickReport2Size		;Joystick report size

SendJoystickReport:
		ldi		temp0, JoystickReportID		; flip report ID
		eor		JoystickFlags, temp0

		rcall CheckAxisRanges

	; simulate call to AddCRCOut
	; simply push point of return onto stack
		ldi		temp0, low(AddCRCOutReturn)	;ROMpointer to descriptor
		push	temp0
		ldi		temp0,  high(AddCRCOutReturn)
		push	temp0
		
		
		ldi	USBBufptrY,JoystickBufferBegin
		ldi	ByteCount,2		;length of output buffer (only SOP and PID)
		add	ByteCount,temp3		;+ number of bytes
		push	USBBufptrY
		push	ByteCount
		rjmp	AddCRCOut_2		;addition of CRC to buffer


ReadButtonsRow:
		nop							; for synchronization
		in		temp0,PIND
		andi	temp0,0b01111011 	;mask out PD2 and PD7 which is INT0 and Column 4
		in		temp1,PINB
		andi	temp1,0b00000100	;extract bit PB2
		or 		temp0,temp1			;merge values
		com		temp0
		andi	temp0,0b01111111
		ret



;*********************************************************************		
;* Start the ADC conversion cycle
;*
;* Input: 	r18 	- number of ADC channel
;*
;* Registers used:
;*			r18
;*********************************************************************
StartADC10: 
		andi r18, 0b00000111
		sts	ADCChanStarted,r18	; set last ADC  channel number
		ori	 r18, 0b01000000	;AVCC refernce, clear ADLAR for 10 bit conversion
		out ADMUX, r18
		sbi		ADCSRA, ADSC	; start conversion
		ret

;*********************************************************************		
;* Read 10bit unsigned value from ADC
;*
;* Input: 	r16 	- number of ADC channel
;*
;* Output:	r17:r16	- 10bit ADC value
;* Registers used:
;*			r16, r17
;*********************************************************************
ReadADC10: 
		lds	r17, ADCChanStarted
		cp	r16, r17
		breq	WaitForADC10	; if conversion was started earlier then go and wait for the result
		andi r16, 0b00000111	; otherwise start the conversion
		ori	 r16, 0b01000000	;AVCC refernce, clear ADLAR for 10 bit conversion
		out ADMUX, r16
		sbi		ADCSRA, ADSC	; start conversion
WaitForADC10:
		sbic	ADCSRA, ADSC
		rjmp	WaitForADC10
		ldi r16, 0xFF
		sts	ADCChanStarted,r16	; reset last ADC  channel number
		in	r16,ADCL
		in	r17,ADCH
		ret






;*********************************************************************		
;* Initialize 8 bit asymmetrical Axis data from EEPROM
;*
;* Input: 	Z 		- address of 8 bit asymmetric axis structure
;*			Y	 	- EEPROM address
;*********************************************************************
Init8AsymAxis:

		; Reset Update Counter	
		ldi	r16,0
		std Z + s_UpdateCntL, r16
		std Z + s_UpdateCntH, r16
		lds	r16, EEPROMFlags
		tst	r16
		brne	LoadDefaults8

		; ADC Min
		movw	r19:r18, ZH:ZL 	; backup structure pointer
		movw	r21:r20, YH:YL 	; backup structure pointer
		adiw 	ZH:ZL, s_EEPROMStructStart	; point to start od EEPROM part
		ldi	r17, Asym8EEStructSize
		call LoadStructFromEEPROM

		movw ZH:ZL, r19:r18		; restore structure pointer
		movw YH:YL, r21:r20		; restore structure pointer

		; check if eeprom is valid
		ldd r16, Z + s_MaxH
		cpi	r16, 4
		brlo	RetInit8
		; out of range
		; load range defaults 
LoadDefaults8:
		ldi	r16, 0
		std Z + s_MaxL, r16
		std Z + s_MaxH, r16
		ldi r16, 3
		std Z + s_MinL, r16
		ldi r16, 0xFF
		std Z + s_MinH, r16

; Calculate Initial Coeficient
RetInit8:
		call CalcCoef
		ret




;*********************************************************************		
;* Initialize 10 bit symmetrical Axis data from EEPROM
;*
;* Input: 	R17:R16	- 10bit unsigned ADC input value (center)
;*			Z 		- address of 10 bit symmetric axis structure
;*			Y	 - EEPROM address
;*********************************************************************
Init10SymAxis:
		; store ADC Center value
		std Z + s_CenterH, r17
		std Z + s_CenterL, r16

		; Reset Update Counter	
		ldi	r16,0
		std Z + s_UpdateCntL, r16
		std Z + s_UpdateCntH, r16
		lds	r16, EEPROMFlags
		tst	r16
		brne	LoadDefaults10
	
		; ADC Min
		movw	r19:r18, ZH:ZL 	; backup structure pointer
		movw	r21:r20, YH:YL 	; backup structure pointer
		adiw 	ZH:ZL, s_EEPROMStructStart	; point to start od EEPROM part
		ldi	r17, Sym10EEStructSize
		call LoadStructFromEEPROM
		movw ZH:ZL, r19:r18		; restore structure pointer
		movw YH:YL, r21:r20		; restore structure pointer

		; check if eeprom is valid
		ldd r16, Z + s_MaxH
		cpi	r16, 4
		brlo	ContInit10
		; out of range

LoadDefaults10:
		; load range defaults 
		ldi	r16, 0
		std Z + s_MaxL, r16
		std Z + s_MaxH, r16
		ldi r16, 3
		std Z + s_MinL, r16
		ldi r16, 0xFF
		std Z + s_MinH, r16


ContInit10:

		ldd	r17, Z + s_CenterH
		ldd r16, Z + s_CenterL

		; Check if  Center is in the range and update accordingly
		; chek Lower Margin
		ldd r19, Z + s_MinH
		ldd r18, Z + s_MinL
		cp	r16, r18			; check if value exceeds  MINIMUM range
		cpc	r17, r19
		brsh	Init10Check_Max
		std Z + s_MinH, r17
		std Z + s_MinL, r16
		ldi r17, 0				; mark that update was made
		ldi	r16, 1
		std Z + s_UpdateCntH, r17
		std Z + s_UpdateCntL, r16


Init10Check_Max:
		ldd r19, Z + s_MaxH
		ldd r18, Z + s_MaxL
		cp	r18, r16		; check if value exceeds MAXIMUM  range
		cpc	r19, r17
		brsh	Init10Check_End
		std Z + s_MaxH, r17
		std Z + s_MaxL, r16
		ldi r17, 0				; mark that update was made
		ldi	r16, 1
		std Z + s_UpdateCntH, r17
		std Z + s_UpdateCntL, r16

		;call CalcPosCoef


Init10Check_End:


; Calculate Initial Coeficients
		call CalcNegCoef
		call CalcPosCoef
		ret

;************************************************************************
; Process 10 bit symmetric axis ADC value scaling it by appropriate coeficients
;
; Input: 	Z - address of 10 bit symmetric axis structure  (size = 10)
; 			r17:r16 - 10 bit unsigned value from ADC
; Output; 	r17:r16 - 10 bit signed value
;************************************************************************
Process10SymAxis:


; Step 1 : Check Min and Max ranges. If the value exceeds any of them update it 
									;and recalculate coeficients
		ldd r19, Z + s_MinH
		ldd r18, Z + s_MinL
		cp	r16, r18			; check if value exceeds  MINIMUM range
		cpc	r17, r19
		brsh	Check_Max
		std Z + s_MinH, r17
		std Z + s_MinL, r16
		sts  ADCValueH, r17 ; backup ADC value
		sts  ADCValueL, r16
		ldi r17, 0				; mark that update was made
		ldi	r16, 1
		std Z + s_UpdateCntH, r17
		std Z + s_UpdateCntL, r16
		call CalcNegCoef
		lds r17, ADCValueH	; restore ADC value
		lds r16, ADCValueL


Check_Max:
		ldd r19, Z + s_MaxH
		ldd r18, Z + s_MaxL
		cp	r18, r16		; check if value exceeds MAXIMUM  range
		cpc	r19, r17
		brsh	Check_End
		std Z + s_MaxH, r17
		std Z + s_MaxL, r16
		sts  ADCValueH, r17 ; backup ADC value
		sts  ADCValueL, r16
		ldi r17, 0				; mark that update was made
		ldi	r16, 1
		std Z + s_UpdateCntH, r17
		std Z + s_UpdateCntL, r16
		call CalcPosCoef
		lds r17, ADCValueH	; restore ADC value
		lds r16, ADCValueL

Check_End:
		
; Step 2 : Offset
		ldd r19, Z + s_CenterH
		ldd r18, Z + s_CenterL
		sub r16, r18		; subtract 16-bit center offset value
		sbc r17, r19


; Step 3 : Multiply by coeficient
		movw r23:r22, r17:r16
		sbrs r23,7
		rjmp TakePosCoef
		sbr	r25,128		; mark negative range

		; negate the number
		com	r22
		com r23
		subi r22, $FF
		sbci r23, $FF

		ldd r21, Z + s_NegCoefH	; negative range coeficient
		ldd r20, Z + s_NegCoefL
		rjmp MulCoef

TakePosCoef:
		cbr	r25,128		; mark positive range
		ldd r21, Z + s_PosCoefH	;positive range coeficient
		ldd r20, Z + s_PosCoefL

MulCoef:				
		call mul16x16_16

; Step 4: Divide by 128
		lsl r16		
		rol r17
		rol r18
		mov r16, r17
		mov r17, r18	; the result is in r17:r16 register
		andi	r17, 1
		sbrs r25,7			;check if value was negative
		rjmp Scaling_End

		; negate the number
		com	r16
		com r17
		subi r16, $FF
		sbci r17, $FF
		andi r17, $01		; make it 10bit signed value
		sbr	r17, 2
		
Scaling_End:

		; result is in r17:r16
		ret 


;************************************************************************
; Process 8 bit asymmetric axis ADC value scaling it by appropriate coeficients
;
; Input: 	Z - address of 8 bit ssymmetric axis structure (size = 6)
; 			r17:r16 - 10 bit unsigned value from ADC
; Output; 	r16 - 8 bit signed value
;************************************************************************
Process8AsymAxis:


; Step 1 : Check Min and Max ranges. If the value exceeds any of them update it 
									;and recalculate coeficients
		cbr	r20,	1 			; clear update bit
		ldd r19, Z + s_MinH
		ldd r18, Z + s_MinL
		cp	r16, r18			; check if value exceeds  MINIMUM range
		cpc	r17, r19
		brsh	Check_Max8
		std Z + s_MinH, r17
		std Z + s_MinL, r16
		sbr	r20, 	1		; update required



Check_Max8:
		ldd r19, Z + s_MaxH
		ldd r18, Z + s_MaxL
		cp	r18, r16		; check if value exceeds MAXIMUM  range
		cpc	r19, r17
		brsh	Check_End8
		std Z + s_MaxH, r17
		std Z + s_MaxL, r16
		sbr	r20, 	1		; update required

Check_End8:
		sbrs	r20,0		; skip recalculation if update is not required
		rjmp	Proc8Offset
		sts  ADCValueH, r17 ; backup ADC value
		sts  ADCValueL, r16
		ldi r17, 0				; mark that update was made
		ldi	r16, 1
		std Z + s_UpdateCntH, r17
		std Z + s_UpdateCntL, r16
		call CalcCoef
		lds r17, ADCValueH	; restore ADC value
		lds r16, ADCValueL
		
; Step 2 : Offset

Proc8Offset:
		ldd r19, Z + s_MinH
		ldd r18, Z + s_MinL
		sub r16, r18		; subtract 16-bit minimum value
		sbc r17, r19


; Step 3 : Multiply by coeficient
		movw r23:r22, r17:r16
		ldd r21, Z + s_CoefH	; range coeficient
		ldd r20, Z + s_CoefL
		movw	r19:r18, r1:r0	; backup r1, r0
		call mul16x16_16
		mov r16, r17	; the result is in r16 register
		movw	r1:r0, r19:r18	; restore r1, r0
		; result is 8 bit signed value in r16 register
		subi	r16, 128
		ret 


;************************************************************************
; Process 10 bit asymmetric axis ADC value scaling it by appropriate coeficients
;
; Input: 	Z - address of 8 bit ssymmetric axis structure (size = 6)
; 			r17:r16 - 10 bit unsigned value from ADC
; Output; 	r17:r16 - 10 bit signed value
;************************************************************************
Process10AsymAxis:


; Step 1 : Check Min and Max ranges. If the value exceeds any of them update it 
									;and recalculate coeficients
		cbr	r20,	1 			; clear update bit
		ldd r19, Z + s_MinH
		ldd r18, Z + s_MinL
		cp	r16, r18			; check if value exceeds  MINIMUM range
		cpc	r17, r19
		brsh	Check_MaxA10
		std Z + s_MinH, r17
		std Z + s_MinL, r16
		sbr	r20, 	1		; update required

Check_MaxA10:
		ldd r19, Z + s_MaxH
		ldd r18, Z + s_MaxL
		cp	r18, r16		; check if value exceeds MAXIMUM  range
		cpc	r19, r17
		brsh	Check_EndA10
		std Z + s_MaxH, r17
		std Z + s_MaxL, r16
		sbr	r20, 	1		; update required

Check_EndA10:
		sbrs	r20,0		; skip recalculation if update is not required
		rjmp	ProcA10Offset
		sts  ADCValueH, r17 ; backup ADC value
		sts  ADCValueL, r16
		ldi r17, 0				; mark that update was made
		ldi	r16, 1
		std Z + s_UpdateCntH, r17
		std Z + s_UpdateCntL, r16
		call CalcCoef
		lds r17, ADCValueH	; restore ADC value
		lds r16, ADCValueL
		
; Step 2 : Offset

ProcA10Offset:
		ldd r19, Z + s_MinH
		ldd r18, Z + s_MinL
		sub r16, r18		; subtract 16-bit minimum value
		sbc r17, r19


; Step 3 : Multiply by coeficient
		movw r23:r22, r17:r16
		ldd r21, Z + s_CoefH	; range coeficient
		ldd r20, Z + s_CoefL
		movw	r19:r18, r1:r0	; backup r1, r0
		call mul16x16_16
		movw	r1:r0, r19:r18	; restore r1, r0
		
; Step 4: Divide by 128
		
		lsl r16		
		rol r17
		rol r18
		lsl r16		
		rol r17
		rol r18
		mov r16, r17
		mov r17, r18	; the result is in r17:r16 register
		andi	r17, 3
		subi	r16, 0
		sbci	r17, 2
		andi r17, 3
					; the result is in r17:r16 register
		ret 




;*******************************************************************
; Calculate 10 bit symmetrical axis negative coeficient
;
; Input: Z - address of 10 bit symmetric axis structure
;*******************************************************************
CalcNegCoef:
		; Calculate Negative range Coeficient
		ldd r19, Z + s_CenterH
		ldd r18, Z + s_CenterL
		ldd r17, Z + s_MinH
		ldd r16, Z + s_MinL
		sub r18, r16
		sbc r19, r17	; the result in r19:r18 is negative range width
		;---------- check for zero range
		brne	ProceedCalcNegCoef
		;-- if zero just make a coeficient = 1, which produces 0 value on end output
		ldi r17, 0
		ldi	r16, 1
		std Z + s_NegCoefH, r17
		std Z + s_NegCoefL, r16
		ret
		
ProceedCalcNegCoef:		

		ldi r17, 255
		ldi r16, 255
		movw	r21:r20, r15:r14	; backup r14 and r15
		call div16u
		movw	r15:r14, r21:r20 	; restore r14 and r15
		std Z + s_NegCoefH, r17
		std Z + s_NegCoefL, r16
		ret
;*******************************************************************
; Calculate 10 bit symmetric axis positive coeficient
;
; Input: Z - address of 10 bit symmetric axis structure
;*******************************************************************
CalcPosCoef:
		; Calculate Positive range Coeficient  
		ldd r19, Z + s_MaxH
		ldd r18, Z + s_MaxL
		ldd r17, Z + s_CenterH
		ldd r16, Z + s_CenterL
		sub r18, r16
		sbc r19, r17	; the result in r19:r18 is positive range width

		;---------- check for zero range
		brne	ProceedCalcPosCoef
		;-- if zero just make a coeficient = 1, which produces 0 value on end output
		ldi r17, 0
		ldi	r16, 1
		std Z + s_PosCoefH, r17
		std Z + s_PosCoefL, r16
		ret

ProceedCalcPosCoef:		
		ldi r17, 255
		ldi r16, 112
		movw	r21:r20, r15:r14	; backup r14 and r15
		call div16u
		movw	r15:r14, r21:r20 	; restore r14 and r15
		std Z + s_PosCoefH, r17
		std Z + s_PosCoefL, r16

		ret


;*******************************************************************
; Calculate 8 bit ssymmetric axis coeficient
;
; Input: Z - address of 8 bit asymmetric axis structure
;*******************************************************************
CalcCoef:
		; Calculate Positive range Coeficient  
		ldd r19, Z + s_MaxH
		ldd r18, Z + s_MaxL
		ldd r17, Z + s_MinH
		ldd r16, Z + s_MinL
		sub r18, r16
		sbc r19, r17	; the result in r19:r18 is a range width
		;---------- check for zero range
		brne	ProceedCalcCoef
		;-- if zero just make a coeficient = 1, which produces 0 value on end output
		ldi r17, 0
		ldi	r16, 1
		std Z + s_CoefH, r17
		std Z + s_CoefL, r16
		ret



ProceedCalcCoef:		
		ldi r17, 255
		ldi r16, 128;255

		movw	r21:r20, r15:r14	; backup r14 and r15
		call div16u
		movw	r15:r14, r21:r20 	; restore r14 and r15

		std Z + s_CoefH, r17
		std Z + s_CoefL, r16

		ret




;***********************************************************************

ReadADC_10: ; Input:	temp0 contains number of ADC input to read
			; Return:	temp0 contains bits 0-7
			;			temp1 contains bits 8-9
		andi temp0, 0b00000111
		ori	 temp0, 0b01000000	;AVCC refernce, clear ADLAR for 10 bit conversion
		out ADMUX, temp0
		sbi		ADCSRA, ADSC	; start conversion
WaitForADC_10:
		sbic	ADCSRA, ADSC
		rjmp	WaitForADC_10
		in	temp0,ADCL
		in	temp1,ADCH

		cpi	temp0,0
		brne	Subtract512
		cpi	temp1,0
		brne	Subtract512
		ldi	temp0,1
Subtract512:
	; subtract 512 to get values in range (-511, 511)
		subi	temp0, 0x00
		sbci	temp1, 0x02
		andi	temp1, 0x03

		ret


;************************************************************************

ReadADC_8: ; temp0 contains number of ADC input to read
; Read value from ADC X channel
		andi temp0, 0b00000111
		ori	 temp0, 0b01100000	;AVCC refernce, set ADLAR for 8 bit conversion
		out ADMUX, temp0
		sbi		ADCSRA, ADSC	; start conversion
WaitForADC_8:
		sbic	ADCSRA, ADSC
		rjmp	WaitForADC_8

		in	temp0,ADCH

Subtract128:
	; subtract 128 to get values in range (-127, 127)
		subi	temp0, 0x80
		ret

;***************************************************88

BackupRegisters:
		sts	RegistersBackup+0, r0
		sts	RegistersBackup+1, r1
		sts	RegistersBackup+2, r2
		sts	RegistersBackup+3, r3
		sts	RegistersBackup+4, r4
		sts	RegistersBackup+5, r5
		sts	RegistersBackup+6, r6
		sts	RegistersBackup+7, r7
		sts	RegistersBackup+8, r8
		sts	RegistersBackup+9, r9
		sts	RegistersBackup+10, r10
		sts	RegistersBackup+11, r11
		sts	RegistersBackup+12, r12
		sts	RegistersBackup+13, r13
		sts	RegistersBackup+14, r14
		sts	RegistersBackup+15, r15
		sts	RegistersBackup+16, r16
		sts	RegistersBackup+17, r17
		sts	RegistersBackup+18, r18
		sts	RegistersBackup+19, r19
		sts	RegistersBackup+20, r20
		sts	RegistersBackup+21, r21
		sts	RegistersBackup+22, r22
		sts	RegistersBackup+23, r23
		sts	RegistersBackup+24, r24
		sts	RegistersBackup+25, r25
		sts	RegistersBackup+26, r26
		sts	RegistersBackup+27, r27
		sts	RegistersBackup+28, r28
		sts	RegistersBackup+29, r29
		sts	RegistersBackup+30, r30
		sts	RegistersBackup+31, r31
		ret

RestoreRegisters:
		lds	r0,RegistersBackup+0
		lds	r1, RegistersBackup+1
		lds	r2, RegistersBackup+2
		lds	r3, RegistersBackup+3
		lds	r4, RegistersBackup+4
		lds	r5, RegistersBackup+5
		lds	r6, RegistersBackup+6
		lds	r7, RegistersBackup+7
		lds	r8, RegistersBackup+8
		lds	r9, RegistersBackup+9
		lds	r10, RegistersBackup+10
		lds	r11, RegistersBackup+11
		lds	r12, RegistersBackup+12
		lds	r13, RegistersBackup+13
		lds	r14, RegistersBackup+14
		lds	r15, RegistersBackup+15
		lds	r16, RegistersBackup+16
		lds	r17, RegistersBackup+17
		lds	r18, RegistersBackup+18
		lds	r19, RegistersBackup+19
		lds	r20, RegistersBackup+20
		lds	r21, RegistersBackup+21
		lds	r22, RegistersBackup+22
		lds	r23, RegistersBackup+23
		lds	r24, RegistersBackup+24
		lds	r25, RegistersBackup+25
		lds	r26, RegistersBackup+26
		lds	r27, RegistersBackup+27
		lds	r28, RegistersBackup+28
		lds	r29, RegistersBackup+29
		lds	r30, RegistersBackup+30
		lds	r31, RegistersBackup+31
		ret

BackupUpperRegisters:
		sts	RegistersBackup+14, r14
		sts	RegistersBackup+15, r15
		sts	RegistersBackup+16, r16
		sts	RegistersBackup+17, r17
		sts	RegistersBackup+18, r18
		sts	RegistersBackup+19, r19
		sts	RegistersBackup+20, r20
		sts	RegistersBackup+21, r21
		sts	RegistersBackup+22, r22
		sts	RegistersBackup+23, r23
		sts	RegistersBackup+24, r24
		sts	RegistersBackup+25, r25
		sts	RegistersBackup+26, r26
		sts	RegistersBackup+27, r27
		sts	RegistersBackup+28, r28
		sts	RegistersBackup+29, r29
		sts	RegistersBackup+30, r30
		sts	RegistersBackup+31, r31
		ret

RestoreUpperRegisters:
		lds	r14, RegistersBackup+14
		lds	r15, RegistersBackup+15
		lds	r16, RegistersBackup+16
		lds	r17, RegistersBackup+17
		lds	r18, RegistersBackup+18
		lds	r19, RegistersBackup+19
		lds	r20, RegistersBackup+20
		lds	r21, RegistersBackup+21
		lds	r22, RegistersBackup+22
		lds	r23, RegistersBackup+23
		lds	r24, RegistersBackup+24
		lds	r25, RegistersBackup+25
		lds	r26, RegistersBackup+26
		lds	r27, RegistersBackup+27
		lds	r28, RegistersBackup+28
		lds	r29, RegistersBackup+29
		lds	r30, RegistersBackup+30
		lds	r31, RegistersBackup+31
		ret

;******************************************************************************
;*
;* FUNCTION
;*	mul16x16_16
;* DECRIPTION
;*	Multiply of two 16bits numbers with 16bits result.
;* USAGE
;*	r17:r16 = r23:r22 * r21:r20
;* STATISTICS
;*	Cycles :	9 + ret
;*	Words :		6 + ret
;*	Register usage: r0, r1 and r16 to r23 (8 registers)
;* NOTE
;*	Full orthogonality i.e. any register pair can be used as long as
;*	the result and the two operands does not share register pairs.
;*	The routine is non-destructive to the operands.
;*
;******************************************************************************

mul16x16_16:
	mul	r22, r20		; al * bl
	movw	r17:r16, r1:r0
	mul	r23, r20		; ah * bl
	add	r17, r0
	mul	r21, r22		; bh * al
	add	r17, r0
	ret


AddCRCOutReturn:	
		inc	ByteCount		;length of output buffer + CRC16
		inc	ByteCount


	; Backup Control pipe buffer pointers to save Control pipe state
		mov	temp0, OutputBufferLength
		sts BkpOutputBufferLength,temp0
		mov temp0, OutBitStuffNumber
		sts BkpOutBitStuffNumber,temp0


		inc	BitStuffInOut			;transmitting buffer - insertion of bitstuff bits
		ldi	USBBufptrY,JoystickBufferBegin	;to transmitting buffer
		rcall	BitStuff
;		mov	OutputBufferLength,ByteCount	;length of answer store for transmiting
		clr	BitStuffInOut			;receiving buffer - deletion of bitstuff bits


	; copy to Joystick buffer
		sts JoyOutputBufferLength, ByteCount
		sts JoyOutBitStuffNumber, OutBitStuffNumber


	; Restore Control pipe buffer pointers
		lds		temp0, BkpOutputBufferLength
		mov		OutputBufferLength, temp0
		lds		temp0, BkpOutBitStuffNumber
		mov		 OutBitStuffNumber,temp0


	; set joystick data ready flag
		ldi		temp0,JoystickDataReady
		or		JoystickFlags,temp0 
;TestpointEnd -------------
	;	rjmp	CRCReturnPoint

		ret
	;	rjmp	Main


;********************************************************************
;*  Main program END
;********************************************************************
;------------------------------------------------------------------------------------------
;********************************************************************
;*  Interrupt0 interrupt handler
;********************************************************************
INT0Handler:					;interrupt INT0
		in	backupSREG,SREG
		push	temp0
		push	temp1

		ldi	temp0,3			;counter of duration log0
		ldi	temp1,2			;counter of duration log1
		;waiting for begin packet
CheckchangeMinus:
		sbis	inputport,DATAminus	;waiting till change D- to 1
		rjmp	CheckchangeMinus
CheckchangePlus:
		sbis	inputport,DATAplus	;waiting till change D+ to 1
		rjmp	CheckchangePlus
DetectSOPEnd:
		sbis	inputport,DATAplus
		rjmp	Increment0		;D+ =0
Increment1:
		ldi	temp0,3			;counter of duration log0
		dec	temp1			;how many cycles takes log1
		nop
		breq	USBBeginPacket		;if this is end of SOP - receive packet
		rjmp	DetectSOPEnd
Increment0:
		ldi	temp1,2			;counter of duration log1
		dec	temp0			;how many cycles take log0
		nop
		brne	DetectSOPEnd		;if there isn't SOF - continue
		rjmp	EndInt0HandlerPOP2
EndInt0Handler:
		pop	ACC
		pop	R26
		pop	temp3
		pop	temp2
EndInt0HandlerPOP:
		pop	USBBufptrY
		pop	ByteCount
		mov	bitcount,backupbitcount	;restore bitcount register
EndInt0HandlerPOP2:
		pop	temp1
		pop	temp0
		out	SREG,backupSREG
		ldi	shiftbuf,1<<INTF0	;zero interruptu flag INTF0
		out	GIFR,shiftbuf
		reti				;otherwise finish (was only SOF - every millisecond)

USBBeginPacket:
		mov	backupbitcount,bitcount	;backup bitcount register
		in	shiftbuf,inputport	;if yes load it as zero bit directly to shift register
USBloopBegin:
		push	ByteCount		;additional backup of registers (save of time)
		push	USBBufptrY
		ldi	bitcount,6		;initialization of bits counter in byte
		ldi	ByteCount,MAXUSBBYTES	;initialization of max number of received bytes in packet
		ldi	USBBufptrY,InputShiftBufferBegin	;set the input buffer
USBloop1_6:
		in	inputbuf,inputport
		cbr	inputbuf,USBpinmask	;unmask low 2 bits
		breq	USBloopEnd		;if they are zeros - end of USB packet
		ror	inputbuf		;transfer Data+ to shift register
		rol	shiftbuf
		dec	bitcount		;decrement bits counter
		brne	USBloop1_6		;if it isn't zero - repeat filling of shift register
		nop				;otherwise is necessary copy shift register to buffer
USBloop7:
		in	inputbuf,inputport
		cbr	inputbuf,USBpinmask	;unmask low 2 bits
		breq	USBloopEnd		;if they are zeros - end of USB packet
		ror	inputbuf		;transfer Data+ to shift register
		rol	shiftbuf
		ldi	bitcount,7		;initialization of bits counter in byte
		st	Y+,shiftbuf		;copy shift register into buffer and increment pointer to buffer
USBloop0:					;and start receiving next byte
		in	shiftbuf,inputport	;zero bit directly to shift register
		cbr	shiftbuf,USBpinmask	;unmask low 2 bits
		breq	USBloopEnd		;if they are zeros - end of USB packet
		dec	bitcount		;decrement bits counter
		nop				;
		dec	ByteCount		;if not reached maximum buffer
		brne	USBloop1_6		;then receive next

		rjmp	EndInt0HandlerPOP	;otherwise repeat back from begin

USBloopEnd:
		cpi	USBBufptrY,InputShiftBufferBegin+3	;if at least 3 byte not received
		brcs	EndInt0HandlerPOP	;then finish
		lds	temp0,InputShiftBufferBegin+0	;identifier of packet to temp0
		lds	temp1,InputShiftBufferBegin+1	;address to temp1
		brne	TestDataPacket		;if is length different from 3 - then this can be only DataPaket
TestIOPacket:
	
		andi	temp1,0xFE		;MMM mask out bit 0 of address to avoid conflict with endpoint 1
	
		cp	temp1,MyAddress		;if this isn't assigned (address) for me 
		brne	TestDataPacket		;then this can be still DataPacket
TestSetupPacket:			;test to SETUP packet
		cpi	temp0,nNRZISETUPPID
		brne	TestOutPacket		;if this isn't Setup PID - decode other packet
		ldi	State,SetupState
		rjmp	EndInt0HandlerPOP	;if this is Setup PID - receive consecutive Data packet
TestOutPacket:				;test for OUT packet
		cpi	temp0,nNRZIOUTPID
		brne	TestInPacket		;if this isn't Out PID - decode other packet
		ldi	State,OutState
		rjmp	EndInt0HandlerPOP	;if this is Out PID - receive consecutive Data packet
TestInPacket:				;test on IN packet
		cpi	temp0,nNRZIINPID
		brne	TestDataPacket		;if this isn't In PID - decode other packet
		rjmp	AnswerToInRequest
TestDataPacket:				;	test for DATA0 and DATA1 packet
		cpi	temp0,nNRZIDATA0PID
		breq	Data0Packet		;if this isn't Data0 PID - decode other packet
		cpi	temp0,nNRZIDATA1PID
		brne	NoMyPacked		;if this isn't Data1 PID - decode other packet
Data0Packet:
		cpi	State,SetupState	;if was state Setup
		breq	ReceiveSetupData	;receive it
		cpi	State,OutState		;if was state Out
		breq	ReceiveOutData		;receive it
NoMyPacked:
		ldi	State,BaseState		;zero state
		rjmp	EndInt0HandlerPOP	;and receive consecutive Data packet

AnswerToInRequest:
		push	temp2			;backup next registers and continue
		push	temp3
		push	R26
		push	ACC


; this might be  Endpoint1 interrupt query
		lds		temp1,InputShiftBufferBegin+1	;address to temp1
		lds		temp2,InputShiftBufferBegin+2	;endpoint and CRC to temp2
		
		ror		temp1 ; move bit 0 to carry
		ror		temp2 ; bring bit 7 to carry
		swap	temp2
		sbrs	temp1, 0	; check bit 1 (6) of address
		rjmp	AddrBit6Zero
		com		temp2
AddrBit6Zero:
		andi	temp2, 0x0F
		cpi		temp2, 0x0A
		breq	ProcessEndpoint0
		cpi		temp2, 0x05
		breq	ProcessEndpoint1
		rjmp 	EndInt0Handler


ProcessEndpoint0:
		cpi		ActionFlag,DoReadySendAnswer	;if isn't prepared answer
		brne	NoReadySend		;then send NAK
		rcall	SendPreparedUSBAnswer	;transmitting answer back
		and		MyUpdatedAddress,MyUpdatedAddress	;if is MyUpdatedAddress nonzero
		brne	SetMyNewUSBAddress_2	;then is necessary to change USB address
		ldi		State,InState
		ldi		ActionFlag,DoPrepareOutContinuousBuffer
		rjmp	EndInt0Handler		;and repeat - wait for next response from USB
ReceiveSetupData:
		push	temp2			;backup next registers and continue
		push	temp3
		push	R26
		push	ACC
		rcall	SendACK			;accept Setup Data packet
		rcall	FinishReceiving		;finish receiving
		ldi		ActionFlag,DoReceiveSetupData
		rjmp	EndInt0Handler
ReceiveOutData:
		push	temp2			;backup next registers and continue
		push	temp3
		push	R26
		push	ACC
		cpi		ActionFlag,DoReceiveSetupData	;if is currently in process command Setup
		breq	NoReadySend		;then send NAK
		rcall	SendACK			;accept Out packet
		clr		ActionFlag
		rjmp	EndInt0Handler
NoReadySend:
		rcall	SendNAK			;still I am not ready to answer
		rjmp	EndInt0Handler		;and repeat - wait for next response from USB

SetMyNewUSBAddress_2:
		rjmp SetMyNewUSBAddress








;*******************************************************************************************
;*
;*	 ENDPOINT 1
;*
;*******************************************************************************************

ProcessEndpoint1:		; on Endpoint1 In we have interrupt handler which is sending reports of joystick data


	; Check if we have joystick data ready
		sbrs	JoystickFlags, JoystickDataReadyBit
		rjmp	NoJoystickDataReady

	; Backup Control pipe buffer pointers to save Control pipe state
		mov	temp0, OutputBufferLength
		sts BkpOutputBufferLength,temp0

		mov temp0, OutBitStuffNumber
		sts BkpOutBitStuffNumber,temp0



	; Retrieve Joystick buffer parameters
		lds 	temp0,JoyOutBitStuffNumber
		mov 	OutBitStuffNumber, temp0

		lds		ByteCount,JoyOutputBufferLength		;length of answer
		ldi		USBBufptrY,JoystickBufferBegin		;pointer to start of transmition buffer
		rcall	SendUSBBuffer	


	; Restore Control pipe buffer pointers
		lds		temp0, BkpOutputBufferLength
		mov		OutputBufferLength, temp0
		lds		temp0, BkpOutBitStuffNumber
		mov		 OutBitStuffNumber,temp0

		; flip data PID
		lds		temp0, JoystickBufferBegin + 1
		cpi		temp0, DATA0PID
		breq	FlipToDATA1PID
		ldi		temp0, DATA0PID
		rjmp	FlipDone
FlipToDATA1PID:
		ldi		temp0, DATA1PID
FlipDone:
		sts		JoystickBufferBegin + 1,temp0

		ldi		temp0,JoystickDataRequest
		or		JoystickFlags,temp0 ;JoystickDataRequest	; request new joystick data

		rjmp	EndInt0Handler		;and complete

NoJoystickDataReady:
	;  	rcall	SendNAK			;still I am not ready to answer
  	  	rjmp	EndInt0Handler		;and repeat - wait for next response from USB
	

;--END--ENDPOINT1





;------------------------------------------------------------------------------------------
SetMyNewUSBAddress:		;set new USB address in NRZI coded
		clr	MyAddress		;original answer state - of my nNRZI USB address
		ldi	temp2,0b00000001	;mask for xoring
		ldi	temp3,8			;bits counter
SetMyNewUSBAddressLoop:
		mov	temp0,MyAddress		;remember final answer
		ror	MyUpdatedAddress	;to carry transmitting bit LSB (in direction firstly LSB then MSB)
		brcs	NoXORBit		;if one - don't change state
		eor	temp0,temp2		;otherwise state will be changed according to last bit of answer
NoXORBit:
		ror	temp0			;last bit of changed answer to carry
		rol	MyAddress		;and from carry to final answer to the LSB place (and reverse LSB and MSB order)
		dec	temp3			;decrement bits counter
		brne	SetMyNewUSBAddressLoop	;if bits counter isn't zero repeat transmitting with next bit
		clr	MyUpdatedAddress	;zero addresses as flag of its next unchanging
		
		; mask out bit 0 to avoid conflict with endpoints
		mov	temp2, MyAddress
		andi	temp2,0xFE
		mov MyAddress, temp2

		rjmp	EndInt0Handler
;------------------------------------------------------------------------------------------
FinishReceiving:		;corrective actions for receive termination
		cpi	bitcount,7		;transfer to buffer also last not completed byte
		breq	NoRemainingBits		;if were all bytes transfered, then nothing transfer
		inc	bitcount
ShiftRemainingBits:
		rol	shiftbuf		;shift remaining not completed bits on right position
		dec	bitcount
		brne	ShiftRemainingBits
		st	Y+,shiftbuf		;and copy shift register bo buffer - not completed byte
NoRemainingBits:
		mov	ByteCount,USBBufptrY
		subi	ByteCount,InputShiftBufferBegin-1	;in ByteCount is number of received bytes (including not completed bytes)

		mov	InputBufferLength,ByteCount		;and save for use in main program
		ldi	USBBufptrY,InputShiftBufferBegin	;pointer to begin of receiving shift buffer
		ldi	R26,InputBufferBegin+1		;data buffer (leave out SOP)
		push	XH					;save RS232BufptrX Hi index
		clr	XH
MoveDataBuffer:
		ld	temp0,Y+
		st	X+,temp0
		dec	ByteCount
		brne	MoveDataBuffer

		pop	XH					;restore RS232BufptrX Hi index
		ldi	ByteCount,nNRZISOPbyte
		sts	InputBufferBegin,ByteCount		;like received SOP - it is not copied from shift buffer
		ret
;------------------------------------------------------------------------------------------
;********************************************************************
;*  Other procedures
;********************************************************************
;------------------------------------------------------------------------------------------
USBReset:		;iinitialization of USB state engine
		ldi	temp0,nNRZIADDR0	;initialization of USB address
		mov	MyAddress,temp0
		clr	State			;initialization of state engine
		clr	BitStuffInOut
		clr	OutBitStuffNumber
		clr	ActionFlag
		clr	RAMread			;will be reading from ROM
		sts	ConfigByte,RAMread	;unconfigured state
		ret
;------------------------------------------------------------------------------------------
SendPreparedUSBAnswer:	;transmitting by NRZI coding OUT buffer with length OutputBufferLength to USB
		mov	ByteCount,OutputBufferLength		;length of answer
SendUSBAnswer:	;transmitting by NRZI coding OUT buffer to USB
		ldi	USBBufptrY,OutputBufferBegin		;pointer to begin of transmitting buffer
SendUSBBuffer:	;transmitting by NRZI coding given buffer to USB
		ldi	temp1,0			;incrementing pointer (temporary variable)
		mov	temp3,ByteCount		;byte counter: temp3 = ByteCount
		ldi	temp2,0b00000011	;mask for xoring
		ld	inputbuf,Y+		;load first byte to inputbuf and increment pointer to buffer
						;USB as output:
		cbi	outputport,DATAplus	;down DATAPLUS : idle state of USB port
		sbi	outputport,DATAminus	;set DATAMINUS : idle state of USB port
		sbi	USBdirection,DATAplus	;DATAPLUS as output
		sbi	USBdirection,DATAminus	;DATAMINUS as output

		in	temp0,outputport	;idle state of USB port to temp0
SendUSBAnswerLoop:
		ldi	bitcount,7		;bits counter
SendUSBAnswerByteLoop:
		nop				;delay because timing
		ror	inputbuf		;to carry transmiting bit (in direction first LSB then MSB)
		brcs	NoXORSend		;if that it is one - don't change USB state
		eor	temp0,temp2		;iotherwise state will be changed
NoXORSend:
		out	outputport,temp0	;send out to USB
		dec	bitcount		;decrement bits counter - according to carry flag
		brne	SendUSBAnswerByteLoop	; if bits counter isn't zero - repeat transmiting with next bit
		sbrs	inputbuf,0		;if is transmiting bit one - don't change USB state
		eor	temp0,temp2		;otherwise state will be changed
NoXORSendLSB:
		dec	temp3			;decrement bytes counter
		ld	inputbuf,Y+		;load next byte and increment pointer to buffer
		out	outputport,temp0	;transmit to USB
		brne	SendUSBAnswerLoop	;repeat for all buffer (till temp3=0)

		mov	bitcount,OutBitStuffNumber	;bits counter for bitstuff
		cpi	bitcount,0		;if not be needed bitstuff
		breq	ZeroBitStuf
SendUSBAnswerBitstuffLoop:
		ror	inputbuf		;to carry transmiting bit (in direction first LSB then MSB)
		brcs	NoXORBitstuffSend	;if is one - don't change state on USB
		eor	temp0,temp2		;otherwise state will be changed
NoXORBitstuffSend:
		out	outputport,temp0	;transmit to USB
		nop				;delay because of timing
		dec	bitcount		;decrement bits counter - according to carry flag
		brne	SendUSBAnswerBitstuffLoop	;if bits counter isn't zero - repeat transmiting with next bit
		ld	inputbuf,Y		;delay 2 cycle
ZeroBitStuf:
		nop				;delay 1 cycle
		cbr	temp0,3
		out	outputport,temp0	;transmit EOP on USB
		
		ldi	bitcount,5		;delay counter: EOP shouls exists 2 bits (16 cycle at 12MHz)
SendUSBWaitEOP:
		dec	bitcount
		brne	SendUSBWaitEOP
		
		sbi	outputport,DATAminus	;set DATAMINUS : idle state on USB port
		sbi	outputport,DATAminus	;delay 2 cycle: Idle should exists 1 bit (8 cycle at 12MHz)
		cbi	USBdirection,DATAplus	;DATAPLUS as input
		cbi	USBdirection,DATAminus	;DATAMINUS as input
		cbi	outputport,DATAminus	;reset DATAMINUS : the third state on USB port
		ret
;------------------------------------------------------------------------------------------
ToggleDATAPID:
		lds	temp0,OutputBufferBegin+1	;load last PID
		cpi	temp0,DATA1PID			;if last was DATA1PID byte
		ldi	temp0,DATA0PID
		breq	SendData0PID			;then send zero answer with DATA0PID
		ldi	temp0,DATA1PID			;otherwise send zero answer with DATA1PID
SendData0PID:
		sts	OutputBufferBegin+1,temp0	;DATA0PID byte
		ret
;------------------------------------------------------------------------------------------
ComposeZeroDATA1PIDAnswer:
		ldi	temp0,DATA0PID			;DATA0 PID - in the next will be toggled to DATA1PID in load descriptor
		sts	OutputBufferBegin+1,temp0	;load to output buffer
ComposeZeroAnswer:
		ldi	temp0,SOPbyte
		sts	OutputBufferBegin+0,temp0	;SOP byte
		rcall	ToggleDATAPID			;change DATAPID
		ldi	temp0,0x00
		sts	OutputBufferBegin+2,temp0	;CRC byte
		sts	OutputBufferBegin+3,temp0	;CRC byte
		ldi	ByteCount,2+2			;length of output buffer (SOP and PID + CRC16)
		ret
;------------------------------------------------------------------------------------------

;------------------------------------------------------------------------------------------
InitJoystickBufffer:

;		ldi	ZH, high(JoystickAnswer<<1)	;ROMpointer to answer
;		ldi	ZL,  low(JoystickAnswer<<1)
		ldi	temp0,JoystickReport1Size				;number of my bytes answers to temp0

		mov TotalBytesToSend, temp0

		ldi	temp0,SOPbyte
		sts	OutputBufferBegin+0,temp0		;SOP byte
		ldi	temp0,DATA0PID
		sts	OutputBufferBegin+1,temp0		;DATA0PID byte


		mov	temp3,TotalBytesToSend	;otherwise send only given number of bytes
		mov ByteCount,TotalBytesToSend;

		ldi	USBBufptrY,OutputBufferBegin+2	;to transmitting buffer
LoadDescriptorFromROM_2:
		lpm				;load from ROM position pointer to R0 <- (ZH:ZL)
		st	Y+,R0			;R0 save to buffer and increment buffer (Y) <- R0, Y++
		adiw	ZH:ZL,1			;increment index to ROM ; Z++
		dec	ByteCount		;till are not all bytes
		brne	LoadDescriptorFromROM_2	;then load next

		ldi	ByteCount,2		;length of output buffer (only SOP and PID)
		add	ByteCount,temp3		;+ number of bytes
	rcall	AddCRCOut		;addition of CRC to buffer
		inc	ByteCount		;length of output buffer + CRC16
		inc	ByteCount


		inc	BitStuffInOut			;transmitting buffer - insertion of bitstuff bits
		ldi	USBBufptrY,OutputBufferBegin	;to transmitting buffer
	rcall	BitStuff
		mov	OutputBufferLength,ByteCount	;length of answer store for transmiting
		clr	BitStuffInOut			;receiving buffer - deletion of bitstuff bits



	; copy to Joystick buffer
		sts JoyOutputBufferLength, OutputBufferLength
		sts JoyOutBitStuffNumber, OutBitStuffNumber

		mov	ByteCount, OutputBufferLength
	;	ldi	ZH, high(OutputBufferBegin<<1)	;pointer to answer
		clr ZH
		ldi	ZL,  OutputBufferBegin
		ldi	USBBufptrY,JoystickBufferBegin	;to transmitting buffer
CopyToJoyBufferLoop:
		ld	R0, Z+				;load from RAM position pointer to R0 <- (ZH:ZL)
		st	Y+,R0			;R0 save to buffer and increment buffer (Y) <- R0, Y++
;		adiw	ZH:ZL,1			;increment index to ROM ; Z++
		dec	ByteCount		;till are not all bytes
		brne	CopyToJoyBufferLoop	;then load next

		ret
;------------------------------------------------------------------------------------------
InitACKBufffer:
		ldi	temp0,SOPbyte
		sts	ACKBufferBegin+0,temp0		;SOP byte
		ldi	temp0,ACKPID
		sts	ACKBufferBegin+1,temp0		;ACKPID byte
		ret
;------------------------------------------------------------------------------------------
SendACK:
		push	USBBufptrY
		push	bitcount
		push	OutBitStuffNumber
		ldi	USBBufptrY,ACKBufferBegin	;pointer to begin of ACK buffer
		ldi	ByteCount,2			;number of transmit bytes (only SOP and ACKPID)
		clr	OutBitStuffNumber
		rcall	SendUSBBuffer
		pop	OutBitStuffNumber
		pop	bitcount
		pop	USBBufptrY
		ret
;------------------------------------------------------------------------------------------
InitNAKBufffer:
		ldi	temp0,SOPbyte
		sts	NAKBufferBegin+0,temp0		;SOP byte
		ldi	temp0,NAKPID
		sts	NAKBufferBegin+1,temp0		;NAKPID byte
		ret
;------------------------------------------------------------------------------------------
SendNAK:
		push	OutBitStuffNumber
		ldi	USBBufptrY,NAKBufferBegin	;pointer to begin of NACK buffer
		ldi	ByteCount,2			;number of transmited bytes (only SOP and NAKPID)
		clr	OutBitStuffNumber
		rcall	SendUSBBuffer
		pop	OutBitStuffNumber
		ret
;------------------------------------------------------------------------------------------
ComposeSTALL:
		ldi	temp0,SOPbyte
		sts	OutputBufferBegin+0,temp0	;SOP byte
		ldi	temp0,STALLPID
		sts	OutputBufferBegin+1,temp0	;STALLPID byte
		ldi	ByteCount,2			;length of output buffer (SOP and PID)
		ret
;------------------------------------------------------------------------------------------
DecodeNRZI:	;encoding of buffer from NRZI code to binary
		push	USBBufptrY		;back up pointer to buffer
		push	ByteCount		;back up length of buffer
		add	ByteCount,USBBufptrY	;end of buffer to ByteCount
		ser	temp0			;to ensure unit carry (in the next rotation)
NRZIloop:
		ror	temp0			;filling carry from previous byte
		ld	temp0,Y			;nload received byte from buffer
		mov	temp2,temp0		;shifted register to one bit to the right and XOR for function of NRZI decoding
		ror	temp2			;carry to most significant digit bit and shift
		eor	temp2,temp0		;NRZI decoding
		com	temp2			;negate
		st	Y+,temp2		;save back as decoded byte and increment pointer to buffer
		cp	USBBufptrY,ByteCount	;if not all bytes
		brne	NRZIloop		;then repeat
		pop	ByteCount		;restore buffer length
		pop	USBBufptrY		;restore pointer to buffer
		ret				;otherwise finish
;------------------------------------------------------------------------------------------
BitStuff:	;removal of bitstuffing in buffer
		clr	temp3			;counter of omitted bits
		clr	lastBitstufNumber	;0xFF to lastBitstufNumber
		dec	lastBitstufNumber
BitStuffRepeat:
		push	USBBufptrY		;back up pointer to buffer
		push	ByteCount		;back up buffer length
		mov	temp1,temp3		;counter of all bits
		ldi	temp0,8			;sum all bits in buffer
SumAllBits:
		add	temp1,temp0
		dec	ByteCount
		brne	SumAllBits
		ldi	temp2,6			;initialize counter of ones
		pop	ByteCount		;restore buffer length
		push	ByteCount		;back up buffer length
		add	ByteCount,USBBufptrY	;end of buffer to ByteCount
		inc	ByteCount		;and for safety increment it with 2 (because of shifting)
		inc	ByteCount
BitStuffLoop:
		ld	temp0,Y			;load received byte from buffer
		ldi	bitcount,8		;bits counter in byte
BitStuffByteLoop:
		ror	temp0			;filling carry from LSB
		brcs	IncrementBitstuff	;if that LSB=0
		ldi	temp2,7			;initialize counter of ones +1 (if was zero)
IncrementBitstuff:
		dec	temp2			;decrement counter of ones (assumption of one bit)
		brne	NeposunBuffer		;if there was not 6 ones together - don't shift buffer
		cp	temp1,lastBitstufNumber	;
		ldi	temp2,6			;initialize counter of ones (if no bitstuffing will be made then must be started again)
		brcc	NeposunBuffer		;if already was made bitstuffing - don't shift buffer

		dec	temp1	;
		mov	lastBitstufNumber,temp1	;remember last position of bitstuffing
		cpi	bitcount,1		;for pointing to 7-th bit (which must be deleted or where to insert zero)
		brne	NoBitcountCorrect
		ldi	bitcount,9	;
		inc	USBBufptrY		;increment pointer to buffer
NoBitcountCorrect:
		dec	bitcount
		bst	BitStuffInOut,0
		brts	CorrectOutBuffer	;if this is Out buffer - increment buffer length
		rcall	ShiftDeleteBuffer	;shift In buffer
		dec	temp3			;decrement counter of omission
		rjmp	CorrectBufferEnd
CorrectOutBuffer:
		rcall	ShiftInsertBuffer	;shift Out buffer
		inc	temp3			;increment counter of omission
CorrectBufferEnd:
		pop	ByteCount		;restore buffer length
		pop	USBBufptrY		;restore pointer to buffer
		rjmp	BitStuffRepeat		;and restart from begin
NeposunBuffer:
		dec	temp1			;if already were all bits
		breq	EndBitStuff		;finish cycle
		dec	bitcount		;decrement bits counter in byte
		brne	BitStuffByteLoop	;if not yet been all bits in byte - go to next bit
						;otherwise load next byte
		inc	USBBufptrY		;increment pointer to buffer
		rjmp	BitStuffLoop		;and repeat
EndBitStuff:
		pop	ByteCount		;restore buffer length
		pop	USBBufptrY		;restore pointer to buffer
		bst	BitStuffInOut,0
		brts	IncrementLength		;if this is Out buffer - increment length of Out buffer
DecrementLength:				;if this is In buffer - decrement length of In buffer
		cpi	temp3,0			;was at least one decrement
		breq	NoChangeByteCount	;if no - don't change buffer length
		dec	ByteCount		;if this is In buffer - decrement buffer length
		subi	temp3,256-8		;if there wasn't above 8 bits over
		brcc	NoChangeByteCount	;then finish
		dec	ByteCount		;otherwise next decrement buffer length
		ret				;and finish
IncrementLength:
		mov	OutBitStuffNumber,temp3	;remember number of bits over
		subi	temp3,8			;if there wasn't above 8 bits over
		brcs	NoChangeByteCount	;then finish
		inc	ByteCount		;otherwise increment buffer length
		mov	OutBitStuffNumber,temp3	;and remember number of bits over (decremented by 8)
NoChangeByteCount:
		ret				;finish
;------------------------------------------------------------------------------------------
ShiftInsertBuffer:	;shift buffer by one bit to right from end till to position: byte-USBBufptrY and bit-bitcount
		mov	temp0,bitcount		;calculation: bitcount= 9-bitcount
		ldi	bitcount,9
		sub	bitcount,temp0		;to bitcount bit position, which is necessary to clear

		ld	temp1,Y			;load byte which still must be shifted from position bitcount
		rol	temp1			;and shift to the left through Carry (transmission from higher byte and LSB to Carry)
		ser	temp2			;FF to mask - temp2
HalfInsertPosuvMask:
		lsl	temp2			;zero to the next low bit of mask
		dec	bitcount		;till not reached boundary of shifting in byte
		brne	HalfInsertPosuvMask
		
		and	temp1,temp2		;unmask that remains only high shifted bits in temp1
		com	temp2			;invert mask
		lsr	temp2			;shift mask to the right - for insertion of zero bit
		ld	temp0,Y			;load byte which must be shifted from position bitcount to temp0
		and	temp0,temp2		;unmask to remains only low non-shifted bits in temp0
		or	temp1,temp0		;and put together shifted and nonshifted part

		ld	temp0,Y			;load byte which must be shifted from position bitcount
		rol	temp0			;and shift it to the left through Carry (to set right Carry for further carry)
		st	Y+,temp1		;and load back modified byte
ShiftInsertBufferLoop:
		cpse	USBBufptrY,ByteCount	;if are not all entire bytes
		rjmp	NoEndShiftInsertBuffer	;then continue
		ret				;otherwise finish
NoEndShiftInsertBuffer:
		ld	temp1,Y			;load byte
		rol	temp1			;and shift to the left through Carry (carry from low byte and LSB to Carry)
		st	Y+,temp1		;and store back
		rjmp	ShiftInsertBufferLoop	;and continue
;------------------------------------------------------------------------------------------
ShiftDeleteBuffer:	;shift buffer one bit to the left from end to position: byte-USBBufptrY and bit-bitcount
		mov	temp0,bitcount		;calculation: bitcount= 9-bitcount
		ldi	bitcount,9
		sub	bitcount,temp0		;to bitcount bit position, which must be shifted
		mov	temp0,USBBufptrY	;backup pointera to buffer
		inc	temp0			;position of completed bytes to temp0
		mov	USBBufptrY,ByteCount	;maximum position to pointer
ShiftDeleteBufferLoop:
		ld	temp1,-Y		;decrement buffer and load byte
		ror	temp1			;and right shift through Carry (carry from higher byte and LSB to Carry)
		st	Y,temp1			;and store back
		cpse	USBBufptrY,temp0	;if there are not all entire bytes
		rjmp	ShiftDeleteBufferLoop	;then continue

		ld	temp1,-Y		;decrement buffer and load byte which must be shifted from position bitcount
		ror	temp1			;and right shift through Carry (carry from higher byte and LSB to Carry)
		ser	temp2			;FF to mask - temp2
HalfDeletePosuvMask:
		dec	bitcount		;till not reached boundary of shifting in byte
		breq	DoneMask
		lsl	temp2			;zero to the next low bit of mask
		rjmp	HalfDeletePosuvMask
DoneMask:
		and	temp1,temp2		;unmask to remain only high shifted bits in temp1
		com	temp2			;invert mask
		ld	temp0,Y			;load byte which must be shifted from position bitcount to temp0
		and	temp0,temp2		;unmask to remain only low nonshifted bits in temp0
		or	temp1,temp0		;and put together shifted and nonshifted part
		st	Y,temp1			;and store back
		ret				;and finish
;------------------------------------------------------------------------------------------
MirrorInBufferBytes:
		push	USBBufptrY
		push	ByteCount
		ldi	USBBufptrY,InputBufferBegin
		rcall	MirrorBufferBytes
		pop	ByteCount
		pop	USBBufptrY
		ret
;------------------------------------------------------------------------------------------
MirrorBufferBytes:
		add	ByteCount,USBBufptrY	;ByteCount points to the end of message 
MirrorBufferloop:
		ld	temp0,Y			;load received byte from buffer
		ldi	temp1,8			;bits counter
MirrorBufferByteLoop:
		ror	temp0			;to carry next least bit
		rol	temp2			;from carry next bit to reverse order 
		dec	temp1			;was already entire byte
		brne	MirrorBufferByteLoop	;if no then repeat next least bit
		st	Y+,temp2		;save back as reversed byte  and increment pointer to buffer
		cp	USBBufptrY,ByteCount	;if not yet been all
		brne	MirrorBufferloop	;then repeat
		ret				;otherwise finish
;------------------------------------------------------------------------------------------
;CheckCRCIn:
;		push	USBBufptrY	;
;		push	ByteCount	;
;		ldi	USBBufptrY,InputBuffercompare	;
;		rcall	CheckCRC	;
;		pop	ByteCount	;
;		pop	USBBufptrY	;
;		ret	;
;------------------------------------------------------------------------------------------

AddCRCOut:
		push	USBBufptrY
		push	ByteCount
		ldi	USBBufptrY,OutputBufferBegin
AddCRCOut_2:
		rcall	CheckCRC
		com	temp0			;negation of CRC
		com	temp1
		st	Y+,temp1		;save CRC to the end of buffer (at first MSB)
		st	Y,temp0			;save CRC to the end of buffer (then LSB)
		dec	USBBufptrY		;pointer to CRC position
		ldi	ByteCount,2		;reverse bits order in 2 bytes CRC
		rcall	MirrorBufferBytes	;reverse bits order in CRC (transmiting CRC - MSB first)
		pop	ByteCount
		pop	USBBufptrY
		ret
;------------------------------------------------------------------------------------------
CheckCRC:	;input: USBBufptrY = begin of message	,ByteCount = length of message
		add	ByteCount,USBBufptrY	;ByteCount points to the end of message 
		inc	USBBufptrY		;set the pointer to message start - omit SOP
		ld	temp0,Y+		;load PID to temp0
						;and set the pointer to start of message - omit also PID
		cpi	temp0,DATA0PID		;if is DATA0 field
		breq	ComputeDATACRC		;compute CRC16
		cpi	temp0,DATA1PID		;if is DATA1 field
		brne	CRC16End		;if no then finish 
ComputeDATACRC:
		ser	temp0			;initialization of remaider LSB to 0xff
		ser	temp1			;initialization of remaider MSB to 0xff
CRC16Loop:
		ld	temp2,Y+		;load message to temp2 and increment pointer to buffer
		ldi	temp3,8			;bits counter in byte - temp3
CRC16LoopByte:
		bst	temp1,7			;to T save MSB of remainder (remainder is only 16 bits - 8 bit of higher byte)
		bld	bitcount,0		;to bitcount LSB save T - of MSB remainder
		eor	bitcount,temp2		;XOR of bit message and bit remainder - in LSB bitcount
		rol	temp0			;shift remainder to the left - low byte (two bytes - through carry)
		rol	temp1			;shift remainder to the left - high byte (two bytes - through carry)
		cbr	temp0,1			;znuluj LSB remains
		lsr	temp2			;shift message to right
		ror	bitcount		;result of XOR bits from LSB to carry
		brcc	CRC16NoXOR		;if is XOR bitmessage and MSB of remainder = 0 , then no XOR
		ldi	bitcount,CRC16poly>>8	;to bitcount CRC polynomial - high byte
		eor	temp1,bitcount		;and make XOR from remains and CRC polynomial - high byte
		ldi	bitcount,CRC16poly	;to bitcount CRC polynomial - low byte
		eor	temp0,bitcount		;and make XOR of remainder and CRC polynomial - low byte
CRC16NoXOR:
		dec	temp3			;were already all bits in byte
		brne	CRC16LoopByte		;unless, then go to next bit
		cp	USBBufptrY,ByteCount	;was already end-of-message 
		brne	CRC16Loop		;unless then repeat
CRC16End:
		ret				;otherwise finish (in temp0 and temp1 is result)
;------------------------------------------------------------------------------------------
LoadDescriptorFromROM:
		lpm				;load from ROM position pointer to R0
		st	Y+,R0			;R0 save to buffer and increment buffer
		adiw	ZH:ZL,1			;increment index to ROM
		dec	ByteCount		;till are not all bytes
		brne	LoadDescriptorFromROM	;then load next
		rjmp	EndFromRAMROM		;otherwise finish
;------------------------------------------------------------------------------------------
LoadDescriptorFromROMZeroInsert:
		lpm				;load from ROM position pointer to R0
		st	Y+,R0			;R0 save to buffer and increment buffer

		bst	RAMread,3		;if bit 3 is one - don't insert zero
		brtc	InsertingZero		;otherwise zero will be inserted
		adiw	ZH:ZL,1			;increment index to ROM
		lpm				;load from ROM position pointer to R0
		st	Y+,R0			;R0 save to buffer and increment buffer
		clt				;and clear
		bld	RAMread,3		;the third bit in RAMread - for to the next zero insertion will be made
		rjmp	InsertingZeroEnd	;and continue
InsertingZero:
		clr	R0			;for insertion of zero
		st	Y+,R0			;zero save to buffer and increment buffer
InsertingZeroEnd:
		adiw	ZH:ZL,1			;increment index to ROM
		subi	ByteCount,2		;till are not all bytes
		brne	LoadDescriptorFromROMZeroInsert	;then load next
		rjmp	EndFromRAMROM		;otherwise finish
;------------------------------------------------------------------------------------------
LoadDescriptorFromSRAM:
		ld	R0,Z			;load from position RAM pointer to R0
		st	Y+,R0			;R0 save to buffer and increment buffer
		adiw	ZH:ZL,1			;increment index to RAM
		dec	ByteCount		;till are not all bytes
		brne	LoadDescriptorFromSRAM	;then load next
		rjmp	EndFromRAMROM		;otherwise finish
;------------------------------------------------------------------------------------------
LoadDescriptorFromEEPROM:
		out	EEARL,ZL		;set the address EEPROM Lo
		out	EEARH,ZH		;set the address EEPROM Hi
		sbi	EECR,EERE		;read EEPROM to register EEDR
		in	R0,EEDR			;load from EEDR to R0
		st	Y+,R0			;R0 save to buffer and increment buffer
		adiw	ZH:ZL,1			;increment index to EEPROM
		dec	ByteCount		;till are not all bytes
		brne	LoadDescriptorFromEEPROM;then load next
		rjmp	EndFromRAMROM		;otherwise finish
;------------------------------------------------------------------------------------------
LoadXXXDescriptor:
		ldi	temp0,SOPbyte			;SOP byte
		sts	OutputBufferBegin,temp0		;to begin of tramsmiting buffer store SOP
		ldi	ByteCount,8			;8 byte store
		ldi	USBBufptrY,OutputBufferBegin+2	;to transmitting buffer

		and	RAMread,RAMread			;if will be reading from RAM or ROM or EEPROM
		brne	FromRAMorEEPROM			;0=ROM,1=RAM,2=EEPROM,4=ROM with zero insertion (string)
FromROM:
		rjmp	LoadDescriptorFromROM		;load descriptor from ROM
FromRAMorEEPROM:
		sbrc	RAMread,2			;if RAMREAD=4
		rjmp	LoadDescriptorFromROMZeroInsert	;read from ROM with zero insertion
		sbrc	RAMread,0			;if RAMREAD=1
		rjmp	LoadDescriptorFromSRAM		;load data from SRAM
		rjmp	LoadDescriptorFromEEPROM	;otherwise read from EEPROM
EndFromRAMROM:
		sbrc	RAMread,7			;if is most significant bit in variable RAMread=1
		clr	RAMread				;clear RAMread
		rcall	ToggleDATAPID			;change DATAPID
		ldi	USBBufptrY,OutputBufferBegin+1	;to transmitting buffer - position of DATA PID
		ret
;------------------------------------------------------------------------------------------
PrepareUSBOutAnswer:	;prepare answer to buffer
		rcall	PrepareUSBAnswer		;prepare answer to buffer
MakeOutBitStuff:
		inc	BitStuffInOut			;transmitting buffer - insertion of bitstuff bits
		ldi	USBBufptrY,OutputBufferBegin	;to transmitting buffer
		rcall	BitStuff
		mov	OutputBufferLength,ByteCount	;length of answer store for transmiting
		clr	BitStuffInOut			;receiving buffer - deletion of bitstuff bits
		ret






;------------------------------------------------------------------------------------------
PrepareUSBAnswer:	;prepare answer to buffer
		clr	RAMread				;zero to RAMread variable - reading from ROM
		lds	temp0,InputBufferBegin+2	;bmRequestType to temp0
		lds	temp1,InputBufferBegin+3	;bRequest to temp1
		cbr	temp0,0b10011111		;if is 5 and 6 bit zero
		brne	CheckVendor			;then this isn't  Vendor Request
		rjmp	StandardRequest			;but this is standard Request
CheckVendor:
		cpi		temp0, 0b01000000
		breq	VendorRequest
		cpi 	temp0, 0b00100000
		breq	ClassRequest

		rjmp	VendorRequest
;--------------------------


;----------------------------------
ClassRequest:
		cpi	temp1,GET_REPORT		;
		breq	ComposeGET_REPORT		;

		cpi	temp1,GET_IDLE		;
		breq	ComposeGET_IDLE		;

		cpi	temp1,GET_PROTOCOL		;
		breq	ComposeGET_PROTOCOL		;

		cpi	temp1,SET_REPORT		;
		breq	ComposeSET_REPORT		;

		cpi	temp1,SET_IDLE		;
		breq	ComposeSET_IDLE		;

		cpi	temp1,SET_PROTOCOL		;
		breq	ComposeSET_PROTOCOL		;

		rjmp	ZeroDATA1Answer			;if that was something unknown, then prepare zero answer

;--------- Class Requests ------------------

ComposeGET_REPORT:				; 
		rjmp	ZeroDATA1Answer
ComposeGET_IDLE:
		rjmp	ZeroDATA1Answer
ComposeGET_PROTOCOL:
		rjmp	ZeroDATA1Answer
ComposeSET_REPORT:
		rjmp	ZeroDATA1Answer
ComposeSET_IDLE:
		rjmp	ZeroDATA1Answer
ComposeSET_PROTOCOL:
		rjmp	ZeroDATA1Answer




;------------------------------------
VendorRequest:

		rjmp	ZeroDATA1Answer			;if that it was something unknown, then prepare zero answer


;----------------------------- USER FUNCTIONS --------------------------------------



;----------------------------- END USER FUNCTIONS ------------------------------------- END USER FUNCTIONS ------------------------------

OneZeroAnswer:		;send single zero
		ldi	temp0,1				;number of my bytes answers to temp0
		rjmp	ComposeGET_STATUS2





;----------------------------- STANDARD USB REQUESTS ---------------------------------- STANDARD USB REQUESTS ------------------------------
StandardRequest:
		cpi	temp1,GET_STATUS		;
		breq	ComposeGET_STATUS		;

		cpi	temp1,CLEAR_FEATURE		;
		breq	ComposeCLEAR_FEATURE		;

		cpi	temp1,SET_FEATURE		;
		breq	ComposeSET_FEATURE		;

		cpi	temp1,SET_ADDRESS		;if to set address
		breq	ComposeSET_ADDRESS		;set the address

		cpi	temp1,GET_DESCRIPTOR		;if requested descriptor
		breq	ComposeGET_DESCRIPTOR		;generate it

		cpi	temp1,SET_DESCRIPTOR		;
		breq	ComposeSET_DESCRIPTOR		;

		cpi	temp1,GET_CONFIGURATION		;
		breq	ComposeGET_CONFIGURATION	;

		cpi	temp1,SET_CONFIGURATION		;
		breq	ComposeSET_CONFIGURATION	;

		cpi	temp1,GET_INTERFACE		;
		breq	ComposeGET_INTERFACE		;

		cpi	temp1,SET_INTERFACE		;
		breq	ComposeSET_INTERFACE		;

		cpi	temp1,SYNCH_FRAME		;
		breq	ComposeSYNCH_FRAME		;
							;if not found known request
		rjmp	ZeroDATA1Answer			;if that was something unknown, then prepare zero answer

ComposeSET_ADDRESS:
		lds	MyUpdatedAddress,InputBufferBegin+4	;new address to MyUpdatedAddress
		rjmp	ZeroDATA1Answer			;send zero answer

ComposeSET_CONFIGURATION:

		lds	temp0,InputBufferBegin+4	;number of configuration to variable ConfigByte
		sts	ConfigByte,temp0		;
ComposeCLEAR_FEATURE:
ComposeSET_FEATURE:
ComposeSET_INTERFACE:
ZeroStringAnswer:
		rjmp	ZeroDATA1Answer			;send zero answer
ComposeGET_STATUS:
TwoZeroAnswer:
		ldi	temp0,2				;number of my bytes answers to temp0
ComposeGET_STATUS2:
		ldi	ZH, high(StatusAnswer<<1)	;ROMpointer to  answer
		ldi	ZL,  low(StatusAnswer<<1)
		rjmp	ComposeEndXXXDescriptor		;and complete
ComposeGET_CONFIGURATION:
		lds	temp0,ConfigByte
		;and	temp0,temp0			;if I am unconfigured
		;breq	OneZeroAnswer			;then send single zero - otherwise send my configuration
		ldi	temp0,1				;number of my bytes answers to temp0
		ldi	ZH, high(ConfigAnswerMinus1<<1)	;ROMpointer to  answer
		ldi	ZL,  low(ConfigAnswerMinus1<<1)+1
		rjmp	ComposeEndXXXDescriptor		;and complete
ComposeGET_INTERFACE:
		ldi	ZH, high(InterfaceAnswer<<1)	;ROMpointer to answer
		ldi	ZL,  low(InterfaceAnswer<<1)
		ldi	temp0,1				;number of my bytes answers to temp0
		rjmp	ComposeEndXXXDescriptor		;and complete
ComposeSYNCH_FRAME:
ComposeSET_DESCRIPTOR:
		rcall	ComposeSTALL
		ret
ComposeGET_DESCRIPTOR:

					; check if we received HID Class Descriptor request
		lds	temp0,InputBufferBegin+2	;bmRequestType to temp0
		cpi temp0, 0b10000001
		breq	ComposeClassDescriptor
			; if not, process standard descriptor requests
		lds	temp1,InputBufferBegin+5	;DescriptorType to temp1
		cpi	temp1,DEVICE			;DeviceDescriptor
		breq	ComposeDeviceDescriptor		;
		cpi	temp1,CONFIGURATION		;ConfigurationDescriptor
		breq	ComposeConfigDescriptor		;
		cpi	temp1,STRING			;StringDeviceDescriptor
		breq	ComposeStringDescriptor		;

		ret
ComposeClassDescriptor:

		lds	temp1,InputBufferBegin+5	;DescriptorType to temp1
		cpi	temp1,CLASS_HID			;HID class descripto
		breq	ComposeHIDClassDescriptor		;
		cpi	temp1,CLASS_Report		;ConfigurationDescriptor
		breq	ComposeReportDescriptor		;
		cpi	temp1,CLASS_Physical			;StringDeviceDescriptor
		breq	ComposePhysicalDescriptor
		ret
				;
ComposeDeviceDescriptor:
		ldi	ZH, high(DeviceDescriptor<<1)	;ROMpointer to descriptor
		ldi	ZL,  low(DeviceDescriptor<<1)
		ldi	temp0,0x12			;number of my bytes answers to temp0
		rjmp	ComposeEndXXXDescriptor		;and completen
ComposeConfigDescriptor:
		ldi	ZH, high(ConfigDescriptor<<1)	;ROMpointer to descriptor
		ldi	ZL,  low(ConfigDescriptor<<1)
		ldi	temp0,9+9+9+7			;number of my bytes answers to temp0
		rjmp	ComposeEndXXXDescriptor		;and complete

ComposeHIDClassDescriptor:	
		ldi	ZH, high(HIDDescriptor<<1)	;ROMpointer to descriptor
		ldi	ZL,  low(HIDDescriptor<<1)
		ldi	temp0,9			;number of my bytes answers to temp0
		rjmp	ComposeEndXXXDescriptor		;and complete

ComposeReportDescriptor	:	;
		ldi	ZH, high(ReportDescriptor<<1)	;ROMpointer to descriptor
		ldi	ZL,  low(ReportDescriptor<<1)
		ldi	temp0,ReportDescriptorSize			;number of my bytes answers to temp0
		rjmp	ComposeEndXXXDescriptor		;and complete

ComposePhysicalDescriptor:
		rjmp	ComposeEndXXXDescriptor		;and complete

ComposeStringDescriptor:
		ldi	temp1,4+8			;if RAMread=4(insert zeros from ROM reading) + 8(behind first byte no load zero)
		mov	RAMread,temp1
		lds	temp1,InputBufferBegin+4	;DescriptorIndex to temp1
		cpi	temp1,0				;LANGID String
		breq	ComposeLangIDString		;
		cpi	temp1,2				;DevNameString
		breq	ComposeDevNameString
		cpi	temp1,3				;NameString
		breq	ComposeNameString
		cpi	temp1,1				;ComposeVendorString
		breq	ComposeVendorString


		rjmp	ZeroStringAnswer		
		rjmp	ZeroDATA1Answer		;if is DescriptorIndex higher than 2 - send zero answer

ComposeVendorString:
		ldi	ZH, high(VendorStringDescriptor<<1)	;ROMpointer to descriptor
		ldi	ZL,  low(VendorStringDescriptor<<1)
		ldi	temp0,(VendorStringDescriptorEnd-VendorStringDescriptor)*4-2	;number of my bytes answers to temp0
		rjmp	ComposeEndXXXDescriptor		;and complete
ComposeDevNameString:
		ldi	ZH, high(DevNameStringDescriptor<<1)	;ROMpointer to descriptor
		ldi	ZL,  low(DevNameStringDescriptor<<1)
		ldi	temp0,(DevNameStringDescriptorEnd-DevNameStringDescriptor)*4-2	;number of my bytes answers to temp0
		rjmp	ComposeEndXXXDescriptor		;and complete
ComposeNameString:
		ldi	ZH, high(NameStringDescriptor<<1)	;ROMpointer to descriptor
		ldi	ZL,  low(NameStringDescriptor<<1)
		ldi	temp0,(NameStringDescriptorEnd-NameStringDescriptor)*4-2	;number of my bytes answers to temp0
		rjmp	ComposeEndXXXDescriptor		;and complete
ComposeLangIDString:
		clr	RAMread
		ldi	ZH, high(LangIDStringDescriptor<<1)	;ROMpointer to descriptor
		ldi	ZL,  low(LangIDStringDescriptor<<1)
		ldi	temp0,(LangIDStringDescriptorEnd-LangIDStringDescriptor)*2;number of my bytes answers to temp0
		rjmp	ComposeEndXXXDescriptor		;and complete


ComposeEndXXXDescriptor:
		lds	TotalBytesToSend,InputBufferBegin+8	;number of requested bytes to TotalBytesToSend
		cp	TotalBytesToSend,temp0			;if not requested more than I can send
		brcs	HostConfigLength		;transmit the requested number
		mov	TotalBytesToSend,temp0		;otherwise send number of my answers
HostConfigLength:
		mov	temp0,TotalBytesToSend		;
		clr	TransmitPart			;zero the number of 8 bytes answers
		andi	temp0,0b00000111		;if is length divisible by 8
		breq	Length8Multiply			;then not count one answer (under 8 byte)
		inc	TransmitPart			;otherwise count it
Length8Multiply:
		mov	temp0,TotalBytesToSend		;
		lsr	temp0				;length of 8 bytes answers will reach
		lsr	temp0				;integer division by 8
		lsr	temp0
		add	TransmitPart,temp0		;and by addition to last non entire 8-bytes to variable TransmitPart
		ldi	temp0,DATA0PID			;DATA0 PID - in the next will be toggled to DATA1PID in load descriptor
		sts	OutputBufferBegin+1,temp0	;store to output buffer
		rjmp	ComposeNextAnswerPart


;------------------------------------------------------------------------------------------
ZeroDATA1Answer:
		rcall	ComposeZeroDATA1PIDAnswer
		ret
;----------------------------- END USB REQUESTS ------

PrepareOutContinuousBuffer:
		rcall	PrepareContinuousBuffer
		rcall	MakeOutBitStuff
		ret
;------------------------------------------------------------------------------------------
PrepareContinuousBuffer:
		mov	temp0,TransmitPart
		cpi	temp0,1
		brne	NextAnswerInBuffer		;if buffer empty
		rcall	ComposeZeroAnswer		;prepare zero answer
		ret
NextAnswerInBuffer:
		dec	TransmitPart			;decrement general length of answer
ComposeNextAnswerPart:
		mov	temp1,TotalBytesToSend	;decrement number of bytes to transmit 
		subi	temp1,8			;is is necessary to send more as 8 byte
		ldi	temp3,8			;if yes - send only 8 byte
		brcc	Nad8Bytov
		mov	temp3,TotalBytesToSend	;otherwise send only given number of bytes
		clr	TransmitPart
		inc	TransmitPart		;and this will be last answer
Nad8Bytov:
		mov	TotalBytesToSend,temp1	;decremented number of bytes to TotalBytesToSend
		rcall	LoadXXXDescriptor
		ldi	ByteCount,2		;length of output buffer (only SOP and PID)
		add	ByteCount,temp3		;+ number of bytes
		rcall	AddCRCOut		;addition of CRC to buffer
		inc	ByteCount		;length of output buffer + CRC16
		inc	ByteCount
		ret				;finish



;***************************************************************************
;*
;* "div16u" - 16/16 Bit Unsigned Division
;*
;* This subroutine divides the two 16-bit numbers 
;* "dd8uH:dd8uL" (dividend) and "dv16uH:dv16uL" (divisor). 
;* The result is placed in "dres16uH:dres16uL" and the remainder in
;* "drem16uH:drem16uL".
;*  
;* Number of words	:196 + return
;* Number of cycles	:148/173/196 (Min/Avg/Max)
;* Low registers used	:2 (drem16uL,drem16uH)
;* High registers used  :4 (dres16uL/dd16uL,dres16uH/dd16uH,dv16uL,dv16uH)
;*
;***************************************************************************

;***** Subroutine Register Variables

.def	drem16uL=r14
.def	drem16uH=r15
.def	dres16uL=r16
.def	dres16uH=r17
.def	dd16uL	=r16
.def	dd16uH	=r17
.def	dv16uL	=r18
.def	dv16uH	=r19

;***** Code

div16u:	clr	drem16uL	;clear remainder Low byte
	sub	drem16uH,drem16uH;clear remainder High byte and carry

	rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_1		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_2		;else
d16u_1:	sec			;    set carry to be shifted into result

d16u_2:	rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_3		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_4		;else
d16u_3:	sec			;    set carry to be shifted into result

d16u_4:	rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_5		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_6		;else
d16u_5:	sec			;    set carry to be shifted into result

d16u_6:	rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_7		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_8		;else
d16u_7:	sec			;    set carry to be shifted into result

d16u_8:	rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_9		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_10		;else
d16u_9:	sec			;    set carry to be shifted into result

d16u_10:rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_11		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_12		;else
d16u_11:sec			;    set carry to be shifted into result

d16u_12:rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_13		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_14		;else
d16u_13:sec			;    set carry to be shifted into result

d16u_14:rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_15		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_16		;else
d16u_15:sec			;    set carry to be shifted into result

d16u_16:rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_17		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_18		;else
d16u_17:	sec			;    set carry to be shifted into result

d16u_18:rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_19		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_20		;else
d16u_19:sec			;    set carry to be shifted into result

d16u_20:rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_21		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_22		;else
d16u_21:sec			;    set carry to be shifted into result

d16u_22:rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_23		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_24		;else
d16u_23:sec			;    set carry to be shifted into result

d16u_24:rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_25		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_26		;else
d16u_25:sec			;    set carry to be shifted into result

d16u_26:rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_27		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_28		;else
d16u_27:sec			;    set carry to be shifted into result

d16u_28:rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_29		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_30		;else
d16u_29:sec			;    set carry to be shifted into result

d16u_30:rol	dd16uL		;shift left dividend
	rol	dd16uH
	rol	drem16uL	;shift dividend into remainder
	rol	drem16uH
	sub	drem16uL,dv16uL	;remainder = remainder - divisor
	sbc	drem16uH,dv16uH	;
	brcc	d16u_31		;if result negative
	add	drem16uL,dv16uL	;    restore remainder
	adc	drem16uH,dv16uH
	clc			;    clear carry to be shifted into result
	rjmp	d16u_32		;else
d16u_31:sec			;    set carry to be shifted into result

d16u_32:rol	dd16uL		;shift left dividend
	rol	dd16uH
	ret



;------------------------------------------------------------------------------------------
.equ	USBversion		=0x0100		;for what version USB is that (1.00)
.equ	VendorUSBID		=0x0000		;vendor identifier 
;********************************************************************************
;*
;*	IMPORTANT NOTE about Vendor ID
;*
;*	For some reason known only to God the possibility to change 
;*	joystick's position in control panel is not implemented 
;*	in Microsoft Windows XP.
;*	All joysticks in control panel and DirectX enumeration appear 
;*	in order of increasing Vendor ID.
;*	So if you want MJoy to appear on top of the list set VendorUSBID=0x0000.
;*	If you want to have second MJoy behaving as secondary 
;*	controls (throttle quadrant, trimmers, propeller pitch, mixture, etc.)
;*	you can set this controller's VendorUSBID=0x0001 and so on.
;*	Of course if you want  to have MJoy the last on the list I suppose you
;*	should set this to VendorUSBID=0xFFFF
;*
;*	Disclaimer: Vendor ID values that are specified in standards and vendors list
;*	are different than that just described. Change this to your own risk although 
;*	I don't see any unless you set it equal to some existing USB device on 
;*	your system. This measure is a workaround to overcome a flaw in operating
;*	system.
;*
;********************************************************************************
.equ	DeviceUSBID		=0x0001		;product identifier (USB Joystick)
.equ	DeviceVersion	=0x0102		;version number of product (version=1.02 - MJoy with autocalibration)
.equ	MaxUSBCurrent	=0xA0		;current consumption from USB (100mA - arbitrary) 
;------------------------------------------------------------------------------------------
DeviceDescriptor:
		.db	0x12,0x01		;0 byte - size of descriptor in byte
						;1 byte - descriptor type: Device descriptor
		.dw	USBversion		;2,3 byte - version USB LSB (1.00)
		.db	0x00,0x00		;4 byte - device class
						;5 byte - subclass
		.db	0x00,0x08		;6 byte - protocol code
						;7 byte - FIFO size in bytes
		.dw	VendorUSBID		;8,9 byte - vendor identifier 
		.dw	DeviceUSBID		;10,11 byte - product identifier 
		.dw	DeviceVersion		;12,13 byte - product version number 
		.db	0x01,0x02		;14 byte - index of string "vendor"
						;15 byte - index of string "product"
		.db	0x00,0x01		;16 byte - index of string "serial number" (0=none)
						;17 byte - number of possible configurations
DeviceDescriptorEnd:
;------------------------------------------------------------------------------------------
ConfigDescriptor:
		.db	0x9,0x02		;length, descriptor type
ConfigDescriptorLength:
		.dw	9+9+9+7			;entire length of all descriptors + HID 
	ConfigAnswerMinus1:			;for sending the number - congiguration number (attention - addition of 1 required)
		.db	1,1			;numInterfaces, congiguration number
		.db	2,0x80			;string index (0=none), attributes; bus powered
;InterfaceDescriptor-1:
		.db	MaxUSBCurrent/2,0x09  ;current consumption,    interface descriptor length
		.db	0x04,0			;interface descriptor; number of interface
	InterfaceAnswer:			;for sending number of alternatively interface
		.db	0,1			;alternatively interface; number of endpoints except EP0
		.db	0x03,0			;interface class - HID; interface subclass
		.db	0,3			;protocol code; string index - Device name
HIDDescriptor:
		.db 0x09,0x21	; HID descriptor length , HID descriptor type (defined by USB)
		.dw 0x101		; HID Class Specification release number
		.db	0,0x01			;Hardware target country.;	;Number of HID class descriptors to follow.
		.db	0x22,ReportDescriptorSize			;Report descriptor type.; length LSB
		.db	0, 0x07			;Total length of Report descriptor MSB, EndPointDescriptor length
;EndPointDescriptor:	
;.db	0x07,0x5		;length, descriptor type - endpoint
		.db	0x5, 0x81		;, descriptor type - endpoint
;.db	0x81,0			;eendpoint address; transfer type
		.db	0x3, 0x08			;endpoint address In 1; transfer type -interrupt;max packet size LSB
;.dw	0x08			;max packet size
		.db	0, 10			;max packet size MSB,polling interval [ms];
;.db	10,0			;polling interval [ms]; dummy byte (for filling)

ConfigDescriptorEnd:
;-------------------------

StatusAnswer:
		.db 0,0		;2 zero answers




.equ	ReportDescriptorSize =(0x51 + 32)
.equ	JoystickReportCount =2

.equ	JoystickReport1Size =8
.equ	JoystickReport2Size =5

;-----------------------------------

ReportDescriptor:
		.db 0x05,0x01		;Usage_Page (Generic Desktop)
		.db 0x15,0x00		;Logical_Minimum (0)
		.db 0x09,0x04		;Usage (Joystick)
		.db 0xA1,0x01		;Collection (Application)
		.db 0x85,0x01			;Report_ID (1)
		.db 0x09,0x01			;Usage (Pointer)
		.db 0xA1,0x00			;Collection (Physical)
		.db 0x09,0x30				;Usage (X) - OK
		.db 0x09,0x31				;Usage (Y) - OK
		.db 0x16,0x00				;Logical_Minimum (-512)
		.db 0xFE,0x26			
		.db 0xFF,0x01				;Logical Maximum (511)
		.db 0x75,0x0A				;Report_Size (10)
		.db 0x95,0x02				;Report_Count (2)
		.db 0x81,0x02				;Input (Data, Var, Abs)
		.db 0x05,0x02			;Usage_Page (Simulation Controls)
		.db 0x09,0xBA			;Usage (Rudder) - OK
		.db 0x09,0xBB			;Usage (Throttle) - OK
		.db 0x81,0x02			;Input (Data, Var, Abs)
		.db 0xC0,0x16				;End_Collection
		.db 0x01,0xFE			;dummy logical minimum
		.db	0x05,0x01			;Usage_Page (Generic Desktop)
		.db 0x09,0x32			;Usage (Slider) - OK
		.db 0x09,0x33			;Usage (Dial) - OK
		.db 0x15,0x81			;Logical_Minimum (-127)
		.db 0x25,0x7F			;Logical Maximum (127)
		.db 0x75,0x08			;Report_Size (8)
		.db 0x95,0x02			;Report_Count (2)
		.db 0x81,0x02			;Input (Data, Var, Abs)
		.db 0x85,0x02			;Report_ID (2)
		.db 0x05,0x09			;Usage_Page (Button)
		.db 0x19,0x01			;Usage_Minimum (Button 1)
		.db 0x29,0x18			;Usage_Maximum (Button 24)
		.db	0x15,0x00			;Logical_Minimum (0)
		.db 0x25,0x01			;Logical_Maximum (1)
		.db 0x75,0x01			;Report_Size (1)
		.db 0x95,0x18			;Report_Count (24)
		.db 0x55,0x00			;Unit_Exponent (0)
		.db 0x65,0x00			;Unit (None)
		.db 0x81,0x02			;Input (Data, Var, Abs)
		.db 0x05,0x01		;Usage_Page (Generic Desktop)
		.db 0x09,0x39			;Usage (Hat switch)
		.db 0x35,0x00			;Physical_Minimum (0)
		.db 0x46,0x3B			;Physical_Maximum (315)
		.db	0x01,0x16				;...  dummy logical minimum
		.db 0x01,0xFE			;
		.db 0x15,0x00			;Logical_Minimum (0)
		.db 0x25,0x07			;Logical_Maximum (7)
		.db 0x65,0x14			;Unit (Eng Rot:Angular Pos)
		.db 0x75,0x04			;Report_Size (4)
		.db 0x95,0x01			;Report_Count (1)
		.db 0x81,0x02			;Input (Data, Var, Abs)
        .db 0x09,0x01           ;USAGE (Vendor Usage 1)
		.db 0x75,0x04			;Report_Size (4)
		.db 0x95,0x01			;Report_Count (1)
		.db 0x81,0x03			;Input (Data, Const, Abs)

		.db 0xC0,0			;End_Collection , dummy padding

ReportDescriptorEnd:


;bits:
; 0 - UP
; 1 - RIGHT
; 2 - DOWN
; 3 - LEFT
HatValues:		; hat values mapping table
		.db 0x0F, 0x00		; 0 = no hat	, 1= Up
		.db 0x02, 0x01		; 2 = Right		, 3 = Up Right
		.db 0x04, 0x0F		; 4 = Down		, 5 = invalid (D U)
		.db 0x03, 0x0F		; 6 = Down Right, 7 = invalid (D U R)
		.db 0x06, 0x07		; 8 = Left		, 9 = Up Left
		.db 0x0F, 0x0F		; 10 = invalid (L R), 11 = invalid (L R U)
		.db 0x05, 0x0F		; 12 = Down Left , invalid
		.db 0x0F, 0x0F		; invalid values



;------------------------------------------------------------------------------------------
LangIDStringDescriptor:
		.db	(LangIDStringDescriptorEnd-LangIDStringDescriptor)*2,3	;length, type: string descriptor
		.dw	0x0009			;English
LangIDStringDescriptorEnd:
;------------------------------------------------------------------------------------------
VendorStringDescriptor:
		.db	(VendorStringDescriptorEnd-VendorStringDescriptor)*4-2,3	;length, type: string descriptor
CopyRight:
		.db	"Mindaugas Milasauskas (c) 2004, Ing. Igor Cesko, Copyright(c) 2003"
CopyRightEnd:
VendorStringDescriptorEnd:
;------------------------------------------------------------------------------------------
DevNameStringDescriptor:
		.db	(DevNameStringDescriptorEnd-DevNameStringDescriptor)*4-2,3;length, type: string descriptor
		.db	"MJoy"
;********************************************************************************
;*
;*	NOTE about DevNameStringDescriptor
;*
;*	This name will appear in game devices list control panel in Windows. 
;*	If you want to have several MJoy devices on the system you can 
;*	change this name together with Vendor ID to the one you like.
;*	Please note that for some reason only first 4 characters will be 
;*	displayed so there is no point making it longer than 4 characters.
;*
;********************************************************************************
DevNameStringDescriptorEnd:

NameStringDescriptor:
		.db	(NameStringDescriptorEnd-NameStringDescriptor)*4-2,3;length, type: string descriptor
		.db	"MJoy, ATMega8 based USB Joystick v1.2"
NameStringDescriptorEnd:
;------------------------------------------------------------------------------------------
;********************************************************************
;*  End of program
;********************************************************************
;------------------------------------------------------------------------------------------
;********************************************************************
;*  EEPROM contents
;********************************************************************
;------------------------------------------------------------------------------------------
.eseg 		;data in EEPROM (at final version comment)
;.org	0x400	;for filling EEPROM give on right addresses - behind the program code (at final version uncomment)
EEData:
;**********************************************************
;* Axis ranges initial information.
;* It is advisory but not necessary to program EEPROM
;* as the software writes correct values when it 
;* detects empty EEPROM.
;**********************************************************
XAxisEE:
	.db 3,255,0,0
YAxisEE:
	.db 3,255,0,0
RudderAxisEE:
	.db 3,255,0,0
ThrottleAxisEE:
	.db 3,255,0,0
ZAxisEE:
	.db 3,255,0,0
RxAxisEE:
	.db 3,255,0,0

;------------------------------------------------------------------------------------------


;********************************************************************
;*  End of file
;********************************************************************


