//****************************************************************************
//*** MAXIMUS V1.0                                                         ***
//*** Hacettepe University 2003 Graduation Project                         ***
//*** Written by Alper YILDIRIM                                            ***
//*** 29.05.2003                                                           ***
//****************************************************************************

#include <16f877.h>     // Header definitions
#include <stdio.h>

#ORG 0x1F00,0x1FFF {}   // For the 8k 16F876/7 protect bootloader code

#fuses HS,NOWDT,NOPROTECT     // Configuration bits
#use delay(clock=20000000)    // Operating frequency
#use rs232(baud=115200, xmit=PIN_C6, rcv=PIN_C7)   //Enable RS-232 communication

#define MODE_X       PIN_D0   // PIC pin definitions
#define STEP_X       PIN_D1
#define DIR_X        PIN_D2
#define ENABLE_X     PIN_D3

#define MODE_Y       PIN_D4
#define STEP_Y       PIN_D5
#define DIR_Y        PIN_D6
#define ENABLE_Y     PIN_D7

#define MODE_Z       PIN_B4
#define STEP_Z       PIN_B5
#define DIR_Z        PIN_B6
#define ENABLE_Z     PIN_B7

#define SPINDLE_BIT  PIN_B2

#define PACKETLENGTH 10       // Define the length of Drill Data Packet
#define STEPSBACKATLIMIT 700  // Move 700 steps in limit switch interrupt

#define ISMACHINEREADY        0x61  // Byte definitions
#define MACHINEREADY          0x62
#define OPENJOGMODE           0x63
#define JOGMODEON             0x64
#define JOGMODEOFF            0x65
#define JOGMODEDONE           0x66
#define OPENDRILLMODE1        0x67
#define DRILLMODE1ON          0x68
#define ONEHOLEDRILLED        0x69
#define DRILLMODE1INITIALIZED 0x6A
#define DRILLMODE1DONE        0x70
#define OPENDRILLMODE2        0x71
#define DRILLMODE2ON          0x72
#define DRILLMODE2OFF         0x73
#define DRILLMODE2DONE        0x74
#define ATOFFSETPOS           0x75
#define EMERGENCYSTOP         0x80
#define MACHINESTOPPED        0x81
#define ERRORLIMITSWITCH      0x90

#define HOME_SWITCH_X   0x01     // Define outgoing bytes for
#define LIMIT_SWITCH_X  0x02     // Home and Limit Switches
#define HOME_SWITCH_Y   0x04
#define LIMIT_SWITCH_Y  0x08
#define HOME_SWITCH_Z   0x10
#define LIMIT_SWITCH_Z  0x20

// Function prototypes
void constspeed(int16 conststep, int8 delay, int8);
void constspeed2(int16 conststep, int8 delay, int8);  // For int_ext routine
void constspeed_xy(int16 StepX, int16 StepY, int8 DelayX, int8 DelayY);
void dly_100us(int8 n);

// Global variables
int8  rs232_buffer;           // Stores the return data from getc() function
int8  dataIn[PACKETLENGTH];   // Stores the incoming data packet from Com Port
int16 jogDataInputLocation;   // Stores the address of the location where interrupts will return
int16 modeSelectionLocation;  // Stores the address of the location where interrupts will return
int1  currentMachineMode;     // MachineMode Flag 0=Jog 1=Drill
int8  axisDelay;              // used in int_ext routine & main()

#int_ext // RB0 External Interrupt Service Routine
void isr_ext()
{
// INTF Flag of INTCON register must be cleared if goto is used in isr
int8 switchConditions;
disable_interrupts(GLOBAL);
delay_ms(300); // debounce button
switchConditions=input_a();
switch (switchConditions)
   {
   case 0x3E:  // RA0=0
      output_bit(DIR_X,1);
      constspeed2(STEPSBACKATLIMIT,axisDelay,1);
      putc(HOME_SWITCH_X);
   break;
   case 0x3D:  // RA1=0
      output_bit(DIR_X,0);
      constspeed2(STEPSBACKATLIMIT,axisDelay,1);
      putc(LIMIT_SWITCH_X);
   break;
   case 0x3B:  // RA2=0
      output_bit(DIR_Y,1);
      constspeed2(STEPSBACKATLIMIT,axisDelay,2);
      putc(HOME_SWITCH_Y);
   break;
   case 0x37:  // RA3=0
      output_bit(DIR_Y,0);
      constspeed2(STEPSBACKATLIMIT,axisDelay,2);
      putc(LIMIT_SWITCH_Y);
   break;
   case 0x2F:  // RA4=0
      putc(HOME_SWITCH_Z);
   break;
   case 0x1F:  // RA5=0
      putc(LIMIT_SWITCH_Z);
   break;
   case 0x3F:  // Unpressed switch Interrupt
      return;
   break;
   default:
      //putc(ERRORLIMITSWITCH);    // On Error send ERRORLIMITSWITCH byte
      return;
   break;
   }
switch (currentMachineMode)
   {
   case 0:  // Jog Mode
      goto_address(jogDataInputLocation);
   break;
   case 1:  // Drill Mode
      goto_address(modeSelectionLocation);
   break;
   }
}  // End of RB0 External Interrupt Service Routine

#int_rda // UART Receive Data Interrupt Service Routine
void serial_isr()
{
rs232_buffer=getc();
switch (rs232_buffer)
   {
   case (EMERGENCYSTOP):
      // Disable all drivers,Cut motor coil currents
      output_bit(ENABLE_X,0);
      output_bit(ENABLE_Y,0);
      output_bit(ENABLE_Z,0);
      putc(MACHINESTOPPED);
      switch (currentMachineMode)
         {
         case 0:  // Jog Mode
            goto_address(jogDataInputLocation);
         break;
         case 1:  // Drill Mode
            goto_address(modeSelectionLocation);
         break;
         }
   break;
   }
}  // End of UART Receive Data Interrupt Service Routine

void main()
{
int16 stepX,stepY,stepZ,drillStepZup,drillStepZdown,axisStep;
int8  delayX,delayY,delayZdown,delayZup,axisSelect,config,jogconfig;
int1  modeX,modeY,modeZ,axisMode,dirX,dirY,dirZ,axisDir,spindle;

output_b(0xF0);   // Set all Driver Pins
output_d(0xFF);
set_tris_b(0x01); // RB0 is input for external interrupts
printf("MAXIMUS_V1.0");
jogDataInputLocation = label_address(JOG_DATA_INPUT);
modeSelectionLocation = label_address(MODE_SELECTION);

disable_interrupts(GLOBAL);   // Disable all Interrputs

while (getc()!=ISMACHINEREADY);   // getc() until ISMACHINEREADY has come
putc(MACHINEREADY);

MODE_SELECTION:
rs232_buffer=getc();

switch (rs232_buffer)
   {
   case (ISMACHINEREADY):
      putc(MACHINEREADY);
      goto MODE_SELECTION;
   break;
   case (OPENJOGMODE):
      putc(JOGMODEON);
      goto JOG_DATA_INPUT;
   break;
   case (OPENDRILLMODE1):
      putc(DRILLMODE1ON);
      goto DRILL_MODE1_INITIALIZE;
   break;
   case (OPENDRILLMODE2):  // !Not Completed!
      putc(DRILLMODE2ON);
      goto JOG_DATA_INPUT; // open jog mode to take 2 hole positions
   break;
   }

JOG_DATA_INPUT:

currentMachineMode = 0; // Machine is in Jog Mode

dataIn[0] = getc(); // Axis Configuration Byte
dataIn[1] = getc(); // Axis Step Low Byte
dataIn[2] = getc(); // Axis Step High Byte
dataIn[3] = getc(); // Axis Delay Byte

bit_clear(*0x0B,1);  // Clear INTF Flag of INTCON register

// if dataIn bit7 = 0 exit jog mode
if ((dataIn[0] >> 6) == 0)
   {
   putc(JOGMODEOFF);
   disable_interrupts(GLOBAL);
   goto MODE_SELECTION;
   }

// else, make the jog movement
axisStep = make16(dataIn[2],dataIn[1]);   // Construct Axis Step Data
axisDelay = dataIn[3];
jogconfig = dataIn[0];

// Construct control bits
axisSelect = (jogconfig & 0xC0) >> 6;     // 01=X, 10=Y, 11=Z
axisMode = jogconfig & 0x20;              // 0=FULL, 1=HALF
axisDir = jogconfig & 0x10;               // 0=CW, 1=CCW
Spindle = jogconfig & 0x01;               // 0=SpindleOFF, 1=SpindleON

// Before movement, enable RS232 and RB0 interrupts for
// Emergency Stop and Axis Limit Switches, respectively
ext_int_edge(H_TO_L);         // RB0 interrupts on falling edge
enable_interrupts(INT_EXT);
enable_interrupts(INT_RDA);
enable_interrupts(GLOBAL);

// Enable all drivers, i.e Energize motor coils
output_bit(ENABLE_X,1);
output_bit(ENABLE_Y,1);
output_bit(ENABLE_Z,1);

output_bit(SPINDLE_BIT,Spindle); // Initialize Spindle Pin

// Do the axis movement according to given parameters
switch (axisSelect)
   {
   case 1:  // 1=01b -> X-Axis
      output_bit(MODE_X,axisMode);
      output_bit(DIR_X,axisDir);
      constspeed(axisStep,axisDelay,1); // (Stepnumber,delay,pin)
   break;
   case 2:  // 2=10b -> Y-Axis
      output_bit(MODE_Y,axisMode);
      output_bit(DIR_Y,axisDir);
      constspeed(axisStep,axisDelay,2); // (Stepnumber,delay,pin)
   break;
   case 3:  // 3=11b -> Z-Axis
      output_bit(MODE_Z,axisMode);
      output_bit(DIR_Z,axisDir);
      constspeed(axisStep,axisDelay,3); // (Stepnumber,delay,pin)
   break;
   }

putc(JOGMODEDONE);      // Inform PC that Movement has been completed
disable_interrupts(GLOBAL);
goto JOG_DATA_INPUT;    // Wait for new coordinates at JOG_DATA_INPUT

DRILL_MODE1_INITIALIZE: // Drill using offset position of the machine

currentMachineMode = 1; // Machine is in Drill Mode

dataIn[0] = getc();  // Movement Configuration Byte
dataIn[1] = getc();  // Step X Low Byte
dataIn[2] = getc();  // Step X High Byte
dataIn[3] = getc();  // Step Y Low Byte
dataIn[4] = getc();  // Step Y High Byte
dataIn[5] = getc();  // Step Z Low Byte
dataIn[6] = getc();  // Step Z High Byte
dataIn[7] = getc();  // Delay X Byte
dataIn[8] = getc();  // Delay Y Byte
dataIn[9] = getc();  // Delay Z Byte

bit_clear(*0x0B,1);  // Clear INTF Flag of INTCON register

// Construct Step and Delay bytes
config = dataIn[0];
StepX = make16(dataIn[2],dataIn[1]);
StepY = make16(dataIn[4],dataIn[3]);
StepZ = make16(dataIn[6],dataIn[5]);
DelayX = dataIn[7];
DelayY = dataIn[8];
DelayZup = dataIn[9];

// Construct Mode and Direction bits
ModeX = config & 0x08;
ModeY = config & 0x10;
ModeZ = config & 0x20;
DirX = config & 0x01;
DirY = config & 0x02;
DirZ = config & 0x04;

// Before movement, enable RS232 and RB0 interrupts for
// Emergency Stop and Axis Limit Switches, respectively
ext_int_edge(H_TO_L);         // RB0 interrupts on falling edge
enable_interrupts(INT_EXT);
enable_interrupts(INT_RDA);
enable_interrupts(GLOBAL);

// Enable all drivers, i.e Energize motor coils
output_bit(ENABLE_X,1);
output_bit(ENABLE_Y,1);
output_bit(ENABLE_Z,1);

// Initialize Mode and Direction pins
output_bit(MODE_X,ModeX);
output_bit(MODE_Y,ModeY);
output_bit(MODE_Z,ModeZ);
output_bit(DIR_X,DirX);
output_bit(DIR_Y,DirY);
output_bit(DIR_Z,DirZ);

// Do the axis movement
constspeed(StepZ,DelayZup,3);    // Z
constspeed(StepX,DelayX,1);      // X
constspeed(StepY,DelayY,2);      // Y

putc(ATOFFSETPOS);   // Inform PC that Offset Movement has been completed

disable_interrupts(GLOBAL);   // Disable Interrupts before taking data packet

// Initialize Drill Mode
dataIn[0] = getc();  // Movement Configuration Byte (ModeX,ModeY,ModeZ)
dataIn[1] = getc();  // drillStepZup Low Byte
dataIn[2] = getc();  // drillStepZup High Byte
dataIn[3] = getc();  // drillStepZdown Low Byte
dataIn[4] = getc();  // drillStepZdown High Byte
dataIn[5] = getc();  // delayX Byte
dataIn[6] = getc();  // delayY Byte
dataIn[7] = getc();  // delayZup Byte
dataIn[8] = getc();  // delayZdown Byte

config = dataIn[0];
drillStepZup = make16(dataIn[2],dataIn[1]);
drillStepZdown = make16(dataIn[4],dataIn[3]);
DelayX = dataIn[5];
DelayY = dataIn[6];
DelayZup = dataIn[7];
DelayZdown = dataIn[8];

ModeX = config & 0x01;
ModeY = config & 0x02;
ModeZ = config & 0x04;

output_bit(MODE_X,ModeX);
output_bit(MODE_Y,ModeY);
output_bit(MODE_Z,ModeZ);

output_bit(DIR_Z,1);                      // Z down
constspeed(drillStepZup,DelayZup,3);      // Z

putc(DRILLMODE1INITIALIZED);  // Inform PC that DrillMode1 has been initialized
// End of Initialize Drill Mode

DRILL_MODE1_INPUT:

dataIn[0] = getc(); // Movement Configuration Byte (Drill?,DirX,DirY)
dataIn[1] = getc(); // drillStepZup Low Byte
dataIn[2] = getc(); // drillStepZup High Byte
dataIn[3] = getc(); // drillStepZdown Low Byte
dataIn[4] = getc(); // drillStepZdown High Byte

if (!(dataIn[0] >> 7))  // If Continue Drilling Flag Reset exit Drill Mode
   {
   putc(DRILLMODE1DONE);   // inform PC that drilling has been completed
   disable_interrupts(GLOBAL);
   goto MODE_SELECTION;    // Goto and Wait at MODE_SELECTION
   }

if ((dataIn[0] & 0x40) >> 6)  // If Start Drilling Flag Set return without drilling
   {
   putc(ONEHOLEDRILLED);   // Machine is ready for taking next hole coordinates
   disable_interrupts(GLOBAL);   // Disable Interrupts before taking data packet
   goto DRILL_MODE1_INPUT; // Wait for new hole coordinates
   }

// else, continue drilling
config = dataIn[0];
StepX = make16(dataIn[2],dataIn[1]);
StepY = make16(dataIn[4],dataIn[3]);

DirX = config & 0x01;
DirY = config & 0x02;

output_bit(DIR_X,DirX);
output_bit(DIR_Y,DirY);

// Before movement, enable RS232 and RB0 interrupts for
// Emergency Stop and Axis Limit Switches, respectively
ext_int_edge(H_TO_L);         // RB0 interrupts on falling edge
enable_interrupts(INT_EXT);
enable_interrupts(INT_RDA);
enable_interrupts(GLOBAL);

// Do the Dual Axis movement
constspeed_xy(StepX,StepY,DelayX,DelayY);

output_bit(DIR_Z,1);                      // Z down
constspeed(drillStepZdown,DelayZdown,3);  // Z
output_bit(DIR_Z,0);                      // Z up
constspeed(drillStepZdown,DelayZup,3);    // Z

putc(ONEHOLEDRILLED);   // Machine is ready for taking next hole coordinates
disable_interrupts(GLOBAL);   // Disable Interrupts before taking data packet

goto DRILL_MODE1_INPUT; // Wait for new hole coordinates

}  // End of main()

void constspeed_xy(int16 StepX, int16 StepY, int8 DelayX, int8 DelayY)
{
int1  axis;
int8  DelayXY;
int16 StepXY,StepMore,i;

if (DelayX < DelayY)    // DelayXY is the speed of slow axis
   DelayXY = DelayY;
else
   DelayXY = DelayX;

if (StepX > StepY)      // StepXY is the step number of the smallest movement
   {
   StepXY = StepY;
   axis = 0;            // axis = X axis
   StepMore = StepX - StepY;
   }
else
   {
   StepXY = StepX;
   axis = 1;            // axis = Y axis
   StepMore = StepY - StepX;
   }

// Do the dual axis movement
for (i=0; i<StepXY; i++)
   {
   output_bit(STEP_X,1);   // StepX pin HIGH
   output_bit(STEP_Y,1);   // StepY pin HIGH
   dly_100us(DelayXY);
   output_bit(STEP_X,0);   // StepX pin LOW
   output_bit(STEP_Y,0);   // StepY pin LOW
   dly_100us(DelayXY);
   }
output_bit(STEP_X,1);      // Add the last step on X axis
output_bit(STEP_Y,1);      // Add the last step on Y axis

// Do the remaining single axis movement
switch (axis)
   {
   case 0:  // X Axis
      for (i=0; i<StepMore; i++)
      {
      output_bit(STEP_X,1);   // Step pin HIGH
      dly_100us(DelayX);
      output_bit(STEP_X,0);   // Step pin LOW
      dly_100us(DelayX);
      }
      output_bit(STEP_X,1);   // Add last step
   break;
   case 1:  // Y Axis
      for (i=0;i<StepMore;i++)
      {
      output_bit(STEP_Y,1);   // Step pin HIGH
      dly_100us(DelayY);
      output_bit(STEP_Y,0);   // Step pin LOW
      dly_100us(DelayY);
      }
      output_bit(STEP_Y,1);   // Add last step
   break;
   }
}  // End of constspeed_xy()

void constspeed(int16 conststep, int8 delay, int8 pin)
// Build conststep number of step pulses
// Frequency = 1/(2*delay*100us)
{
int16 i;

switch (pin) {
   case 1:
      for (i=0;i<conststep;i++)
      {
      output_bit(STEP_X,1);   // Step pin HIGH
      dly_100us(delay);
      output_bit(STEP_X,0);   // Step pin LOW
      dly_100us(delay);
      }
      output_bit(STEP_X,1);   // Add last step
   break;
   case 2:
      for (i=0;i<conststep;i++)
      {
      output_bit(STEP_Y,1);   // Step pin HIGH
      dly_100us(delay);
      output_bit(STEP_Y,0);   // Step pin LOW
      dly_100us(delay);
      }
      output_bit(STEP_Y,1);   // Add last step
   break;
   case 3:
      for (i=0;i<conststep;i++)
      {
      output_bit(STEP_Z,1);   // Step pin HIGH
      dly_100us(delay);
      output_bit(STEP_Z,0);   // Step pin LOW
      dly_100us(delay);
      }
      output_bit(STEP_Z,1);   // Add last step
   break;
   }
}  // End of constspeed()

// Same as constspeed() funtion (used for int_ext routine)
void constspeed2(int16 conststep, int8 delay, int8 pin)
// Build conststep number of step pulses
// Frequency = 1/(2*delay*100us)
{
int16 i;

switch (pin) {
   case 1:
      for (i=0;i<conststep;i++)
      {
      output_bit(STEP_X,1);   // Step pin HIGH
      dly_100us(delay);
      output_bit(STEP_X,0);   // Step pin LOW
      dly_100us(delay);
      }
      output_bit(STEP_X,1);   // Add last step
   break;
   case 2:
      for (i=0;i<conststep;i++)
      {
      output_bit(STEP_Y,1);   // Step pin HIGH
      dly_100us(delay);
      output_bit(STEP_Y,0);   // Step pin LOW
      dly_100us(delay);
      }
      output_bit(STEP_Y,1);   // Add last step
   break;
   case 3:
      for (i=0;i<conststep;i++)
      {
      output_bit(STEP_Z,1);   // Step pin HIGH
      dly_100us(delay);
      output_bit(STEP_Z,0);   // Step pin LOW
      dly_100us(delay);
      }
      output_bit(STEP_Z,1);   // Add last step
   break;
   }
}  // End of constspeed2()

void dly_100us(int8 n)  // 3us+(1.2us+d)*n
{
// Delay for n * 100us
for(;n!=0;n--)
delay_us(98); // d is exactly 98.8 us
delay_cycles(4); // 0.8 us for 20Mhz oscillator
}

